import copy
import datetime
import json
import logging
import math
import os
import platform
import re
import socket
import subprocess
import time
from configparser import ConfigParser

import requests
import requests.exceptions
from lxml import etree

from consts import ROOT_PATH, SITE_NAME, SIWIM_E_PATH, SIWIM_E_VERSION, global_vars
from generic_module import Module


def swmCurSite(loc):
    try:
        cfg = ConfigParser(strict=False)  # Ignore potential duplicate keys in conf
        cfg.read(loc)
        return cfg.get("global", "site")
    except:
        return "generic"


class ReceiveModule(Module):
    def __init__(self, args):
        expected_params = {"typ": None, "host": None, "port": None, "recv_buffer_size": 2048, "save_original": 0, "root_path": None, "offset_from_boss": {}, "validity_tag": "", "vehicle_end": "0", "uname": "", "passw": "", "cypher_name": "",
                           "lane_sep_offset_start": 510, "lane_sep_angle": 73, "left_lane": 2}
        Module.__init__(self, args, expected_params)
        self.s = None
        self.parser = etree.XMLParser(remove_blank_text=True)
        self.am_i_a_server = False
        self.cyphers = dict()
        # Might make sense to pass these two to expected params, but not sure if that'd affect anything. Besides, this should be reworked for v5.
        self.root_path = args[ROOT_PATH]
        self.siwim_e_path = args[SIWIM_E_PATH]
        if self.cypher_name != "":
            self.init_cyphers(self.cypher_name)

    def init_cyphers(self, cypher_name):
        # parse cypher | cypher must be located in conf folder
        try:
            fd = open("conf/" + cypher_name)
            lines = fd.readlines()
            fd.close()
            header_parts = lines[0].split(";")
            for i in range(1, len(header_parts)):
                self.cyphers[header_parts[i].strip()] = dict()
                for j in range(1, len(lines)):
                    line_parts = lines[j].split(";")
                    self.cyphers[header_parts[i].strip()][line_parts[0].strip()] = line_parts[i].strip()
        except:
            return

    def set_end(self):
        self.end = True
        if self.am_i_a_server:
            s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            try:
                s.connect(("127.0.0.1", self.port))
            except Exception as e:  # Should be ConnectionRefused, but Python2 doesn't have it.
                self.logger.critical('Reset failed because {}:{} refused connection.'.format("127.0.0.1", self.port))
                logging.getLogger('siwim_i').critical('Reset failed because {}:{} refused connection:\n{}'.format("127.0.0.1", self.port, e))

    def generate_path(self, mod):
        pth = self.sites_path
        pth = pth + "/" + self.sname + "/ext"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + mod
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + self.typ
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + self.name
        if not os.path.exists(pth):
            os.mkdir(pth)
        return pth

    def save_to_file(self, data_type, str_vehicles):
        date = datetime.datetime.now().strftime('%Y-%m-%d')
        path = self.generate_path(data_type)
        fd = None
        if os.path.exists(path + "/" + date + ".xml"):
            fd = open(path + "/" + date + ".xml", "a+")
        else:
            first_line = '<?xml version="1.0" ?>\n<swd version=' + '"' + "1" + '"><site><name>' + self.sname + '</name><vehicles>'
            fd = open(path + "/" + date + ".xml", "a+")
            fd.write(first_line)
        if fd == None:
            return
        fd.seek(0, os.SEEK_END)
        pos = fd.tell() - 1
        # find the last vehicle closing tag
        while pos > 0 and fd.read(10) != "</vehicle>":
            pos -= 1
            fd.seek(pos, os.SEEK_SET)
            # delete everything from the end of last vehicle closing tag onwards
        if pos > 0:
            fd.seek(pos + 10, os.SEEK_SET)
            fd.truncate()
        for str_vehicle in str_vehicles:
            fd.write("\n" + str_vehicle)
        last_line = "</vehicles></site></swd>"
        fd.write("\n" + last_line)
        fd.close()

    def connect_tcp(self):
        self.s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        try:
            self.s.connect((self.host, self.port))
            self.s.settimeout(2)
            return 0
        except socket.error as e:
            self.logger.warning('Connection to {0}:{1} refused ({2})'.format(self.host, self.port, e))
            return 1
        except:
            self.logger.exception('Failed to connect:')
            return 1

    def bind_udp(self):
        self.s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        try:
            self.s.bind(("", self.port))
            # self.s.settimeout(2)
            return 0
        except:
            self.logger.exception('Failed to bind:')
            return 1

    def bind_tcp(self):
        self.s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        try:
            self.s.bind(('', self.port))
            # self.s.settimeout(2)
            return 0
        except:
            self.logger.exception('Failed to bind:')
            return 1

    def w32get_file_info(self, filename, info):
        from ctypes import windll, byref, create_string_buffer, c_uint, string_at, c_void_p, wstring_at
        import sys
        import array
        mygetfileversionsize = windll.version.GetFileVersionInfoSizeW if sys.version_info[0] == 3 else windll.version.GetFileVersionInfoSizeA
        mygetfileversioninfo = windll.version.GetFileVersionInfoW if sys.version_info[0] == 3 else windll.version.GetFileVersionInfoA
        myverqueryvalue = windll.version.VerQueryValueW if sys.version_info[0] == 3 else windll.version.VerQueryValueA
        mystringat = wstring_at if sys.version_info[0] == 3 else string_at
        size = mygetfileversionsize(filename, None)
        if size == 0:
            return ''
        res = create_string_buffer(size)
        mygetfileversioninfo(filename, None, size, res)
        r = c_void_p(0)
        l = c_uint(0)
        myverqueryvalue(res, '\\VarFileInfo\\Translation', byref(r), byref(l))
        if not l.value:
            return ''
        codepages = array.array('H', string_at(r.value, l.value))
        codepage = tuple(codepages[:2].tolist())
        myverqueryvalue(res, ('\\StringFileInfo\\%04x%04x\\' + info) % codepage, byref(r), byref(l))
        return mystringat(r.value, l.value - 1)

    def siwim_new_xml_receiver(self):
        if self.connect_tcp() == 1:
            self.logger.info('Connection failed.')
            return
        swd_version = "1"
        # Get SiWIM_E version and save it to globally accessible variable.
        if platform.system() == 'Windows':
            global_vars[SIWIM_E_VERSION] = self.w32get_file_info(self.root_path + '/siwim_e/bin/siwim_mcp.exe', "FileVersion")
        elif platform.system() == 'Linux':
            global_vars[SIWIM_E_VERSION] = re.search(r'\d+\.\d+\.\d+', subprocess.check_output(['{}/siwim_e/bin/siwim_mcp'.format(self.root_path), '-h'], timeout=10).decode()).group()
        global_vars[SITE_NAME] = swmCurSite((self.siwim_e_path if self.siwim_e_path else "{}/siwim_e".format(self.root_path)) + "/conf/siwim.conf")
        try:
            for key, mod in self.downstream_modules_dict.items():
                mod.tell_offset(self.name, self.offset_from_boss)
            line = ""
            bogus_count = 0  # for exiting thread
            timeout_count = 0  # for exiting thread
            while True:
                if self.end:
                    return
                received = ""
                try:
                    received = self.s.recv(self.recv_buffer_size).decode(errors='replace')
                    if received == "":
                        self.logger.warning('Received an empty string, exiting thread.')
                        return
                except socket.timeout:
                    self.zzzzz(1.0)
                    continue
                except socket.error:
                    self.logger.debug('{0} with buffer [{1}] closed.'.format(self.s, self.recv_buffer_size))
                    self.logger.warning('Connection closed.')
                    return
                line = line + received
                starting_tag_id = line.find("<swd ")
                closing_tag_id = line.find("</swd>")
                original_xml = None
                if starting_tag_id != -1 and closing_tag_id != -1:
                    dt_str = line[starting_tag_id:closing_tag_id + 6]
                    try:
                        original_xml = etree.fromstring(dt_str, self.parser)
                        swd_version = original_xml.attrib["version"]
                        # save everything that we didn't use for xml
                        line = line[(closing_tag_id + 6):len(line)]
                        bogus_count = 0
                    except:
                        self.logger.warning('Bogus content, clearing the line.')
                        # if data between starting and closing swd tag is "corba", we need to clear the line
                        line = ""
                        continue
                else:
                    # if there was no starting tag inside line, we need to clear the line
                    if starting_tag_id == -1:
                        line = ""
                    # 15 being an arbitrary number; 8 was not enough
                    elif bogus_count == 15:
                        self.logger.info("Exiting thread " + self.name + " due to recieved data's non well-formity, end result being: " + line)
                        return
                    bogus_count += 1
                    continue
                original_vehicles = list(original_xml.iter("vehicle"))
                # a list of stringy vehicles
                str_vehicles = list()
                for original_vehicle in original_vehicles:
                    # use cypher if any
                    clss = original_vehicle.find("wim").find("cls").text
                    for key in self.cyphers.keys():
                        try:
                            key_el = etree.Element(key)
                            key_el.text = self.cyphers[key][clss]
                            original_vehicle.find("wim").append(key_el)
                        except:
                            pass
                    if self.validity_tag != "" and original_vehicle.find("wim").find(self.validity_tag) == None:
                        continue
                    str_vehicles.append(etree.tostring(original_vehicle).decode())
                    for key, mod in self.downstream_modules_dict.items():
                        mod.set_swd_version(swd_version)
                        mod.add_vehicle(copy.deepcopy(original_vehicle), self.name)
                        self.logger.debug("Sending vehicle with ts " + original_vehicle.find("wim").find("ts").text + " to downstream.")
                # save to file if enabled
                if self.save_original == 1:
                    self.save_to_file("wim", str_vehicles)
        except:
            self.logger.exception('XML receiver error:')
            return

    def hash_djb2(self, s):
        hsh = 5381
        for x in s:
            hsh = ((hsh << 5) + hsh) + ord(x)
        msh = ("%x" % (hsh & 0xFFFFFFFF)).upper()
        while len(msh) < 8:
            msh = "0" + msh
        return "<HSH|" + msh + ">"
        # return "<HSH|" + ("%x" % (hsh & 0xFFFFFFFF))[2:-1].upper() + ">"

    def comark_lwh_server(self):
        try:
            self.am_i_a_server = True
            if self.bind_tcp() == 1:
                self.logger.info('Connection failed.')
                return
            self.s.listen(3);
            for key, mod in self.downstream_modules_dict.items():
                mod.tell_offset(self.name, self.offset_from_boss)
            while True:
                conn, addr = self.s.accept()
                conn.settimeout(2.0)
                try:
                    if self.end:
                        try:
                            conn.shutdown()
                        except:
                            pass
                        try:
                            conn.close()
                        except:
                            pass
                        return
                    line = ""
                    cnt = 0  # for exiting thread
                    last_heartbeat = time.time()
                    while True:
                        if self.end:
                            return
                        received = ""
                        try:
                            received = conn.recv(self.recv_buffer_size).decode(errors='replace')
                        except socket.timeout:
                            # if no heartbeat for more than a lot
                            if time.time() - last_heartbeat > 50:
                                self.logger.info('No heartbeat. Restarting.')
                                return
                            continue
                        except socket.error:
                            self.logger.exception('Comark:')
                            return
                        else:
                            line = line + received
                            starting_tag_id = line.find("<sensor ")
                            closing_tag_id = line.find("</sensor>")
                            original_xml = None
                            if starting_tag_id != -1 and closing_tag_id != -1:
                                dt_str = line[starting_tag_id:closing_tag_id + 9]
                                try:
                                    original_xml = etree.fromstring(dt_str, self.parser)
                                    # save everything that we didn't use for xml
                                    hsh = self.hash_djb2(dt_str)
                                    if hsh != "None":
                                        conn.send(hsh.encode())
                                    line = line[(closing_tag_id + 9):len(line)]
                                    cnt = 0
                                except:
                                    # if data between starting and closing swd tag is "corba", we need to clear the line
                                    self.logger.info('Bogus content, clearing the line.')
                                    self.logger.exception('Comark:')
                                    line = ""
                                    continue
                            else:
                                # if there was no starting tag inside line, we need to clear the line
                                if starting_tag_id == -1:
                                    line = ""
                                cnt += 1
                                if cnt == 8:
                                    self.logger.info('Exiting thread: ' + self.name)
                                    return
                                continue
                            # add a container element in case line contains more than one message
                            messages = list(original_xml.iter("sensor"))
                            str_vehicles = list()
                            for message in messages:
                                vehicle = etree.Element("vehicle")
                                vehicle_1 = etree.Element("vehicle")
                                lwh = etree.Element("lwh")
                                vehicle.append(lwh)
                                try:
                                    body = list(message.iter("heartbeat"))[0]
                                    last_heartbeat = time.time()
                                except:
                                    pass
                                try:
                                    body = list(message.iter("transit_end"))[0]
                                    # transit_end is as good as a hertbeat
                                    last_heartbeat = time.time()
                                except:
                                    continue
                                attributes = body.attrib
                                # can't trust this; disregard|nope, we need it
                                """if attributes["direction"] == "N":
                                  continue
                                if attributes["speed"] == "0":
                                  continue"""
                                for key in attributes.keys():
                                    if key == "time_iso_ms":
                                        ts = datetime.datetime.strptime(attributes[key], '%Y-%m-%dT%H:%M:%S.%f').strftime('%Y-%m-%d-%H-%M-%S-%f')
                                        ts = ts[:-3]
                                        lwh_element = etree.Element("ts")
                                        lwh_element.text = ts
                                        lwh.append(lwh_element)
                                    elif key == "speed":
                                        lwh_element = etree.Element("v")
                                        # convert km/h to m/s
                                        lwh_element.text = str(float(attributes[key]) / 3.6)
                                        lwh.append(lwh_element)
                                    elif key == "class_id":
                                        lwh_element = etree.Element("cls")
                                        # convert km/h to m/s
                                        lwh_element.text = str(attributes[key])
                                        lwh.append(lwh_element)
                                        for key1 in self.cyphers.keys():
                                            try:
                                                key_el = etree.Element(key1)
                                                key_el.text = self.cyphers[key1][str(attributes[key])]
                                                lwh.append(key_el)
                                            except Exception as ex:
                                                pass
                                    elif key == "occupancy":
                                        lwh_element = etree.Element(key)
                                        lwh_element.text = attributes[key]
                                        lwh.append(lwh_element)
                                    else:
                                        lwh_element = etree.Element(key)
                                        lwh_element.text = attributes[key]
                                        lwh.append(lwh_element)
                                    # vehicle = etree.Element("vehicle")
                                    vehicle.append(lwh)
                                    vehicle_1.append(body)
                                lwh_element = etree.Element("siwim_i_add_to_offset")
                                lwh_element.text = "0"
                                lwh.append(lwh_element)
                                for key, mod in self.downstream_modules_dict.items():
                                    mod.add_vehicle(copy.deepcopy(vehicle), self.name)
                                str_vehicles.append(etree.tostring(vehicle_1).decode())
                            if self.save_original == 1 and len(str_vehicles) > 0:
                                self.save_to_file("lwh", str_vehicles)
                except:
                    self.logger.exception('Comark:')
                    try:
                        conn.shutdown()
                    except:
                        pass
                    try:
                        conn.close()
                    except:
                        pass
                    return
        except:
            self.logger.exception('Comark:')
            try:
                conn.shutdown()
            except:
                pass
            try:
                conn.close()
            except:
                pass
            return

    def macq_lines_to_dict(self, lines_list):
        lines_dict = dict()
        for line in lines_list:
            line_parts = line.split("=")
            lines_dict[line_parts[0]] = line_parts[1]
        return lines_dict

    def datetime_from_utc_to_local(self, utc_datetime):
        now_timestamp = time.time()
        offset = datetime.datetime.fromtimestamp(now_timestamp) - datetime.datetime.utcfromtimestamp(now_timestamp)
        return utc_datetime + offset

    def macq_anpr_receiver(self):
        if self.bind_udp() == 1:
            self.logger.info('Connection failed.')
            return
        try:
            for key, mod in self.downstream_modules_dict.items():
                mod.tell_offset(self.name, self.offset_from_boss)
            while True:
                if self.end:
                    return
                data, addr = self.s.recvfrom(self.recv_buffer_size)
                data = data.decode(errors='replace')
                data_dict = self.macq_lines_to_dict(data.splitlines())
                # for now; disregard status messages
                if data_dict["type"] == "status":
                    continue
                ts = data_dict["T"][0:4] + "-" + data_dict["T"][4:6] + "-" + data_dict["T"][6:8] + "-" + data_dict["T"][8:10] + "-" + data_dict["T"][10:12] + "-" + data_dict["T"][12:14] + "-" + data_dict["T"][14:17]
                utc = datetime.datetime.strptime(ts, "%Y-%m-%d-%H-%M-%S-%f")
                ts = self.datetime_from_utc_to_local(utc).strftime("%Y-%m-%d-%H-%M-%S-%f")[:-3]
                lp = data_dict["P"]
                hgp = data_dict["D"]
                trg_ip = data_dict["Fc"]
                anpr_ip = data_dict["F"]
                lane = data_dict["lane"]

                vehicle_node = etree.Element("vehicle")
                vehicle_1_node = etree.Element("vehicle")
                anpr_node = etree.Element("anpr")
                anpr_1_node = etree.Element("anpr")

                vehicle_node.append(anpr_node)
                vehicle_1_node.append(anpr_1_node)

                ts_node = etree.Element("ts")
                lp_node = etree.Element("lp")
                hgp_node = etree.Element("hgp")
                anpr_ip_node = etree.Element("anpr_ip")
                trg_ip_node = etree.Element("trg_ip")
                lane_node = etree.Element("lane")

                ts_node.text = ts
                lp_node.text = lp
                hgp_node.text = hgp
                anpr_ip_node.text = anpr_ip
                trg_ip_node.text = trg_ip
                lane_node.text = str(lane)

                anpr_node.append(ts_node)
                anpr_node.append(lp_node)
                anpr_node.append(hgp_node)
                anpr_node.append(anpr_ip_node)
                anpr_node.append(trg_ip_node)
                anpr_node.append(lane_node)
                # print etree.tostring(vehicle_node)
                for key in data_dict.keys():
                    node = etree.Element(key)
                    node.text = data_dict[key]
                    anpr_1_node.append(node)
                for key, mod in self.downstream_modules_dict.items():
                    mod.add_vehicle(copy.deepcopy(vehicle_node), self.name)
                str_vehicles = list()
                str_vehicles.append(etree.tostring(vehicle_1_node).decode())

                if self.save_original == 1 and len(str_vehicles) > 0:
                    self.save_to_file("anpr", str_vehicles)
        except:
            self.logger.exception('Macq error:')
            return

    def cammra_anpr_receiver(self):
        self.am_i_a_server = True
        if self.bind_tcp() == 1:
            self.logger.info('Connection failed.')
            return
        self.s.listen(5);
        for key, mod in self.downstream_modules_dict.items():
            mod.tell_offset(self.name, self.offset_from_boss)
        while True:
            conn, addr = self.s.accept()
            if self.end:
                try:
                    conn.shutdown()
                except:
                    pass
                try:
                    conn.close()
                except:
                    pass
                return
            data = ""
            while True:
                data_dict = None
                try:
                    if self.end:
                        return
                    # clear data if it consists of something that will never be a proper dictionary
                    if len(data) > 2 and data.find("{\"") == -1:
                        data = ""
                    data += conn.recv(self.recv_buffer_size).decode(errors='replace')
                    # cases of an empty string being passed shouldn't occur; but they do
                    if data == "":
                        self.logger.warning("Received an empty string.")
                        try:
                            conn.shutdown()
                        except:
                            pass
                        try:
                            conn.close()
                        except:
                            pass
                        return
                    if "{\"" in data and "\n}" in data and data.find("{\"") < data.find("\n}"):
                        relevant = data[data.find("{\""):data.find("\n}") + 2].encode("utf-8")
                        data_dict = json.loads(relevant, strict=False)
                        try:
                            data_dict.pop('ImageArray')  # remove the image array key from the dictionary. EDIT: why again?
                        except:
                            pass
                        vehicle_node = etree.Element("vehicle")
                        vehicle_1_node = etree.Element("vehicle")
                        anpr_node = etree.Element("anpr")
                        anpr_1_node = etree.Element("anpr")
                        vehicle_node.append(anpr_node)
                        vehicle_1_node.append(anpr_1_node)
                        ts_node = etree.Element("ts")
                        lp_node = etree.Element("lp")
                        trg_ip_node = etree.Element("trg_ip")
                        anpr_ip_node = etree.Element("anpr_ip")
                        cam_addr_node = etree.Element("cam_addr")
                        country_node = etree.Element("country")
                        confidence_node = etree.Element("confidence")
                        axconfig_tiein_node = etree.Element("axconfig_tiein")
                        lp_coordinates_node = etree.Element("lp_coordinates")
                        cam_addr_node.text = addr[0]
                        lane_node = etree.Element("lane")
                        # here, we're gonna try to map roiID (which starts from 1 and so forth) to actual lanes,
                        # based on our offset_from_boss
                        lane_node.text = data_dict["roiID"]
                        if len(self.offset_from_boss) > 0:
                            lane_node.text = str(int(lane_node.text) - 1 + int(min(self.offset_from_boss)))
                            if lane_node.text not in self.offset_from_boss:
                                continue
                        lp_node.text = data_dict["plateUTF8"]
                        ts = data_dict["capture_timestamp"]
                        ms = ts[-3:]
                        ts = datetime.datetime.fromtimestamp(int(ts[:-3])).strftime("%Y-%m-%d-%H-%M-%S") + "-" + ms
                        ts_node.text = ts
                        trg_ip_node.text = data_dict["imagesURI"][1]
                        anpr_ip_node.text = data_dict["imagesURI"][0]
                        location_node = etree.Element("location")
                        location_node.text = "front"
                        try:
                            country_node.text = data_dict["plateCountry"]
                            anpr_node.append(country_node)
                        except:
                            pass
                        try:
                            confidence_node.text = data_dict["plateConfidence"]
                            anpr_node.append(confidence_node)
                        except:
                            pass
                        try:
                            lp_coordinates_node.text = str(data_dict["plateCoordinatesRelative"])
                            anpr_node.append(lp_coordinates_node)
                        except:
                            pass
                        try:
                            vehicle_info = data_dict["vehicle_info"]["type"]
                            if vehicle_info == "BUS":
                                axconfig_tiein_node.text = "B"
                            else:
                                axconfig_tiein_node.text = "X"
                            anpr_node.append(axconfig_tiein_node)
                        except:
                            pass
                        if self.vehicle_end == "1":
                            add_to_offset_node = etree.Element("siwim_i_add_to_offset")
                            add_to_offset_node.text = "0"
                            location_node.text = "rear"
                            anpr_node.append(add_to_offset_node)
                        anpr_node.append(location_node)
                        anpr_node.append(ts_node)
                        anpr_node.append(lp_node)
                        anpr_node.append(lane_node)
                        anpr_node.append(trg_ip_node)
                        anpr_node.append(anpr_ip_node)
                        anpr_node.append(cam_addr_node)
                        # anpr_1_node; save EVERYTHING
                        for key, val in data_dict.items():
                            node = etree.Element(key)
                            try:
                                node.text = str(val)
                            except:
                                continue
                            anpr_1_node.append(node)
                        str_vehicles = list()
                        str_vehicles.append(etree.tostring(vehicle_1_node).decode())
                        for key, mod in self.downstream_modules_dict.items():
                            mod.add_vehicle(copy.deepcopy(vehicle_node), self.name)
                        if self.save_original == 1 and len(str_vehicles) > 0:
                            self.save_to_file("anpr", str_vehicles)
                        data = data[0:data.find("{\"")] + data[data.find("\n}"):-1]
                        try:
                            conn.shutdown()
                        except:
                            pass
                        try:
                            conn.close()
                        except:
                            pass
                        break
                except:  # Log the datetime of the vehicle that had to be ditched.
                    self.logger.exception('Data lost:')
                    try:
                        conn.shutdown()
                    except:
                        pass
                    try:
                        conn.close()
                    except:
                        pass
                    break

    def vidar_downloader(self):
        def parse_resp(resp_text):
            try:
                resp_text = resp_text.replace('<?xml version="1.0" encoding="UTF-8"?>', "")
                parser = etree.XMLParser(remove_blank_text=True)
                xml_resp = etree.fromstring(resp_text, parser=parser)
                id_num = xml_resp.find("ID").attrib["value"]
                ts = xml_resp.find("capture").find("frametime").attrib["value"]
                ts = datetime.datetime.strptime(ts, "%Y.%m.%d %H:%M:%S.%f").strftime('%Y-%m-%d-%H-%M-%S-%f')[:-3]
                anpr = xml_resp.find("anpr").find("text").attrib["value"]
                speed = xml_resp.xpath('trigger/speed/@value')  # This will be either value of speed or empty list if it doesn't exist.
                lprimage = ""
                # try finding photo
                try:
                    images = xml_resp.find("images")
                    lprimage = images.find("lp_img").attrib["value"]
                    # we don't want to save it
                    images.getparent().remove(images)
                except:
                    pass
                try:
                    lane = xml_resp.xpath('trigger/zone_name/@value')[0][1:]
                    if lane == str():
                        raise ValueError('trigger/zone_name/@value is empty')
                    self.logger.debug('Successfully obtained zone_name.')
                except Exception as e:
                    self.logger.warning('No zone_name in data: {}. Obtaining from coords.'.format(e))
                    coords = xml_resp.find("anpr").find("frame").attrib["value"]
                    first_x = float(coords.split(",")[0])
                    first_y = float(coords.split(",")[1])
                    relevant_lanes = sorted(list(self.offset_from_boss.keys()))
                    # default lane is the first lane listed (handling single-lane setups)
                    lane = relevant_lanes[0]
                    # lemove left lane, because it's the one we're going to look for
                    if str(self.left_lane) in relevant_lanes:
                        relevant_lanes.remove(str(self.left_lane))
                        # if additional lanes remain, proceed with magicks
                        if len(relevant_lanes) > 0:
                            # if lane isn't on the left side of the screen, it's on the other
                            lane = relevant_lanes[0]
                            if first_x < (first_y / math.tan(math.radians(self.lane_sep_angle))) + self.lane_sep_offset_start:
                                # otherwise it's the left lane
                                lane = str(self.left_lane)
                if self.save_original == 1:
                    xml_resp.tag = "vehicle"
                    self.save_to_file("anpr", [etree.tostring(xml_resp, pretty_print=False).decode(), ])
                result = {"vidar_id": id_num, "ts": ts, "lp": anpr, "lane": str(lane), "lprimage": lprimage}
                if speed:  # Only add speed if list isn't empty.
                    result['v'] = str(round(float(speed[0]) / 3.6, 2))
                return result
            except:
                # self.logger.exception('Failed to parse Vidar response:')
                return {}

        if self.uname == "" or self.passw == "":
            self.logger.info("ARH credentials not set. Returning.")
            return
        for key, mod in self.downstream_modules_dict.items():
            mod.tell_offset(self.name, self.offset_from_boss)
        login_url = "http://" + self.host + ":" + str(self.port) + "/login.html?p_send=1&p_username=" + self.uname + "&p_passw=" + self.passw
        # WHERE ID> should be followed by previous ID
        result_list_query_url = "http://" + self.host + ":" + str(self.port) + "/lpr/cff?cmd=getresultlist&select=WHERE ID>"
        # first iteration follow id= with last, then with IDs from result list query
        dl_url = "http://" + self.host + ":" + str(self.port) + "/lpr/cff?cmd=getdata&id="
        id_num = "last"
        while True:
            resp_dicts = list()
            with requests.Session() as s:
                try:
                    s.get(login_url, timeout=5)
                except:
                    self.logger.exception("Login timeout:")
                    return
                if id_num == "last":
                    try:
                        resp = s.get(dl_url + id_num, timeout=2)
                    except requests.exceptions.ReadTimeout:
                        self.logger.warning('Read timeout reached for request: {}'.format(dl_url + id_num))
                        return
                    except:
                        self.logger.exception("Data acquisition timeout:")
                        return
                    resp_dict = parse_resp(resp.text)
                    if len(resp_dict) == 0:
                        self.zzzzz(2)
                        continue
                    id_num = resp_dict["vidar_id"]
                    self.logger.debug('Adding {} at {} (lane {}) with plate: {}'.format(resp_dict['vidar_id'], resp_dict['ts'], resp_dict['lane'], resp_dict['lp'].encode()))
                    resp_dicts.append(resp_dict)
                else:
                    # get all new event ids
                    try:
                        resp1 = s.get(result_list_query_url + id_num, timeout=5)
                    except requests.exceptions.ReadTimeout:
                        self.logger.warning('Read timeout reached for request: {}'.format(result_list_query_url + id_num))
                        return
                    except Exception as e:
                        self.logger.warning('An exception has occurred when sending get request. Enable debug logging for stack trace: {}.'.format(e))
                        self.logger.debug('Result list acquisition timeout:', exc_info=True)
                        return
                    xml_resp = etree.fromstring(resp1.text)
                    ids = list()
                    for child in xml_resp.getchildren():
                        if child.tag.startswith("result"):
                            ids.append(child.attrib["value"])
                    ids = sorted(ids)
                    for oneid in ids:
                        id_num = oneid
                        try:
                            resp2 = s.get(dl_url + id_num, timeout=2)
                        except (requests.exceptions.ConnectionError, requests.exceptions.ReadTimeout):
                            self.logger.warning('Read timeout reached or connection aborted for request: {}'.format(dl_url + id_num))
                            return
                        except:
                            self.logger.exception("Data acquisition timeout:")
                            continue
                        resp_dict = parse_resp(resp2.text)
                        if len(resp_dict) == 0:
                            continue
                        self.logger.debug('Adding {} at {} (lane {}) with plate: {}'.format(resp_dict['vidar_id'], resp_dict['ts'], resp_dict['lane'], resp_dict['lp'].encode()))
                        resp_dicts.append(resp_dict)
            for resp_dict in resp_dicts:
                vehicle_node = etree.Element("vehicle")
                anpr_el = etree.Element("anpr")
                for key in resp_dict.keys():
                    key_el = etree.Element(key)
                    key_el.text = resp_dict[key]
                    anpr_el.append(key_el)
                vehicle_node.append(anpr_el)
                # print(etree.tostring(vehicle_node))
                for key, mod in self.downstream_modules_dict.items():
                    mod.add_vehicle(copy.deepcopy(vehicle_node), self.name)
            self.zzzzz(2)

    def technix_anpr_receiver(self):
        if self.connect_tcp() == 1:
            self.logger.info('Connection failed.')
            return
        try:
            for key, mod in self.downstream_modules_dict.items():
                mod.tell_offset(self.name, self.offset_from_boss)
            line = ""
            cnt = 0  # for exiting thread
            while True:
                if self.end:
                    return
                received = ""
                try:
                    received = self.s.recv(self.recv_buffer_size).decode(errors='replace')
                except socket.timeout:
                    continue
                except socket.error:
                    self.logger.exception('Socket error:')
                    return
                else:
                    line = line + received
                    starting_tag_id = line.find("<TAMPxml ")
                    closing_tag_id = line.find("</TAMPxml>")
                    original_xml = None
                    if starting_tag_id != -1 and closing_tag_id != -1:
                        dt_str = line[starting_tag_id:closing_tag_id + 10]
                        try:
                            original_xml = etree.fromstring(dt_str, self.parser)
                            # save everything that we didn't use for xml
                            line = line[(closing_tag_id + 10):len(line)]
                            cnt = 0
                        except:
                            # if data between starting and closing swd tag is "corba", we need to clear the line
                            self.logger.info('Bogus content, clearing the line.')
                            # print line
                            line = ""
                            continue
                    else:
                        # if there was no starting tag inside line, we need to clear the line
                        if starting_tag_id == -1:
                            self.logger.debug("No starting tag, clearing the line.")
                            line = ""
                        cnt += 1
                        if cnt == 8:
                            self.logger.info('Exiting thread: ' + self.name)
                            return
                        continue
                    # add a container element in case line contains more than one message
                    messages = list(original_xml.iter("Message"))
                    str_vehicles = list()
                    for message in messages:
                        if message.attrib["Type"] == "KeepAlive":
                            top_element = etree.Element("TAMPxml", LocationId="TAMPxmlClient")
                            ka_message = etree.Element("Message", Type="KeepAlive")
                            top_element.append(ka_message)
                            self.s.send(etree.tostring(top_element))
                        elif message.attrib["Type"] == "ANPRevent":
                            vehicle = etree.Element("vehicle")
                            vehicle_1 = etree.Element("vehicle")
                            anpr = etree.Element("anpr")
                            vehicle.append(anpr)
                            body = list(message.iter("Body"))[0]
                            attributes = body.attrib
                            kn_num = ""
                            un_num = ""
                            for key in list(attributes.keys()):
                                if key == "TRIGGER_TIME":
                                    first_split = attributes[key].split(" ")
                                    date_split = first_split[0].split(".")
                                    time_split_1 = first_split[1].split(".")
                                    time_split_2 = time_split_1[0].split(":")
                                    ts = date_split[2] + "-" + date_split[1] + "-" + date_split[0] + "-" + time_split_2[0] + "-" + time_split_2[1] + "-" + time_split_2[2] + "-" + time_split_1[1]
                                    anpr_element = etree.Element("ts")
                                    anpr_element.text = ts
                                    anpr.append(anpr_element)
                                elif key == "KN_NUM":
                                    kn_num = attributes[key]
                                elif key == "UN_NUM":
                                    un_num = attributes[key]
                                elif key == "ANPR_TXT":
                                    lp_el = etree.Element("lp")
                                    lp_el.text = attributes[key]
                                    anpr.append(lp_el)
                                elif key == "ANPR_IMAGE_PATH":
                                    anpr_ip_el = etree.Element("anpr_ip")
                                    anpr_ip_el.text = attributes[key]
                                    anpr.append(anpr_ip_el)
                                elif key == "ADR_IMAGE_PATH":
                                    adr_ip_el = etree.Element("adr_ip")
                                    adr_ip_el.text = attributes[key]
                                    anpr.append(adr_ip_el)
                                elif key == "COMBINED_TRG_IMAGE_PATH":
                                    trg_ip_el = etree.Element("trg_ip")
                                    trg_ip_el.text = attributes[key]
                                    anpr.append(trg_ip_el)
                                elif key == "LANE_ID":
                                    lane_el = etree.Element("lane")
                                    # ugly hack! just take the last character, due to a possibility of lane text actually being "Lane1"
                                    # could be something else, but this should cover most cases
                                    lane_el.text = attributes[key][len(attributes[key]) - 1]
                                    anpr.append(lane_el)
                            hdp_el = etree.Element("hgp")
                            hdp_el.text = kn_num + " " + un_num
                            anpr.append(hdp_el)
                            vehicle.append(anpr)
                            vehicle_1.append(body)
                            for key, mod in self.downstream_modules_dict.items():
                                mod.add_vehicle(copy.deepcopy(vehicle), self.name)
                                self.logger.debug("Sending vehicle with ts " + vehicle.find("anpr").find("ts").text + " to downstream.")
                            str_vehicles.append(etree.tostring(vehicle_1).decode())
                    if self.save_original == 1 and len(str_vehicles) > 0:
                        self.save_to_file("anpr", str_vehicles)
        except:
            self.logger.exception('Technix error')
            return

    def run(self):
        self.alive = True
        self.end = False
        # for key, mod in self.downstream_modules_dict.items():
        # mod.register_alarm(self.name, self.typ)
        if self.typ == "technix":
            self.technix_anpr_receiver()
        if self.typ == "cestel":
            self.siwim_new_xml_receiver()
        elif self.typ == "macq":
            self.macq_anpr_receiver()
        # elif self.typ == "comark_client":
        # self.comark_lwh_receiver()
        elif self.typ == "comark":
            self.comark_lwh_server()
        elif self.typ == "ffgroup":
            self.cammra_anpr_receiver()
        elif self.typ == "arh":
            self.vidar_downloader()
        # elif self.typ == "vaxtor":
        # self.vaxtor_anpr_receiver()
        self.alive = False
        try:
            self.s.shutdown()
        except:
            pass
        try:
            self.s.close()
        except:
            pass
