import datetime
import os
import platform
import socket
import subprocess
import time

import serial
import serial.tools.list_ports

from generic_module import Module


class HistModule(Module):
    def __init__(self, args):
        expected_params = {"udp_ip": "127.0.0.1", "udp_port": 9000, "serial_port": "auto", "baud": 9600, "interval": 30, "root_path": None, "relevant_for_listeners": "u", "relevant_for_listeners": "u", "vpn_host": "172.16.0.1", "no_vpn_host_end": 3}
        Module.__init__(self, args, expected_params)
        if self.no_vpn_host_end > 30:
            self.logger.warning('Value no_vpn_host_end has an illogical value. It is recommended to remove it from configuration file.')
            self.no_vpn_host_end = 30
        # automatic detection if applicable
        potential_ports = self.get_hist_ports()
        if self.serial_port == "auto":
            if len(potential_ports) == 1:
                self.serial_port = potential_ports[0]
            else:
                self.logger.critical('Automatic port detection is enabled, but there is more (or less) than one viable port! Please manually pick the correct one: {}'.format(potential_ports))
                self.zzzzz(2)
                return

    def is_in_vpn(self):
        if platform.system() == "Windows":
            ping = subprocess.Popen(["ping", "-n", str(self.no_vpn_host_end), "-w", "2", self.vpn_host], stdout=subprocess.PIPE).communicate()[0].decode()
        elif platform.system() == 'Linux':
            ping = subprocess.check_output(['ping', '-c', str(self.no_vpn_host_end), '-W', '2', self.vpn_host]).decode()
        else:
            raise NotImplementedError('Sending ping is not implemented for {}.'.format(platform.platform()))
        if "ttl=" in ping.lower():
            return True
        return False

    def get_hist_ports(self):
        # only supports windows for now
        usb_serial_ports = list()
        ports = serial.tools.list_ports.comports()  # This returns a list of ListInfoPort objects for PySerial 3.4 (Win10 machines) and generator of tuples for PySerial 2.7 (WinXP machines)
        if platform.system() == "Windows":
            for port in ports:
                # Since PySerial 2.7 is last supported version for Win XP, we have to use its way of getting information for it.
                if platform.release() != 'XP' and "USB Serial Port" in port.description:  # In PySerial 3.4 ports are stored in a ListInfoPort object.
                    usb_serial_ports.append(port.device)
                elif port[1][:15] == "USB Serial Port":  # In PySerial 2.7 ports are simple tuples and have the COM value as part of their name for some reason. TODO Might be worth looking into more detail.
                    usb_serial_ports.append(port[0])
        elif platform.system() == 'Linux':
            for port in ports:
                if port[1] == 'FT230X Basic UART':
                    usb_serial_ports.append(port[0])
        else:
            raise NotImplementedError('Can\'t obtain hist port for {}'.format(platform.platform()))
        return usb_serial_ports

    def generate_path(self, mod):
        pth = self.sites_path
        pth = pth + "/" + self.sname + "/ext"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + mod
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/hysteresis"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + self.name
        if not os.path.exists(pth):
            os.mkdir(pth)
        return pth

    def histcmd(self, ser, cmnd, resp='', to=0.5):
        if ser.timeout != to:
            ser.timeout = to
        ser.flushInput()
        ser.write((cmnd + '\n').encode())
        r = []
        while True:
            l = ser.readline().rstrip().decode()
            if l == '':  # EOF
                return r
            r.append(l)
            if len(resp) > 0 and l.find(resp) != -1 or len(r) > 10:  # resp found or safeguard
                return r

    def dict2confstr(self, section, d):
        r = '[' + section + ']' + '\n'
        for k, v in d.items():
            r += k + '=' + v + '\n'
        return r

    def run(self):
        self.alive = True
        self.end = False
        try:
            ser = serial.Serial(self.serial_port, self.baud)
            try:
                while True:
                    # if server connection
                    if self.no_vpn_host_end > 0 and not self.is_in_vpn():
                        self.logger.info('Resetting due to no connection with VPN IP: {}'.format(self.vpn_host))
                        self.end = True
                    if self.end:
                        ser.close()
                        self.alive = False
                        self.logger.debug('Thread closed correctly.')
                        return
                    d = {'ts': time.strftime('%Y-%m-%d-%H-%M-%S')}
                    r = self.histcmd(ser, 'STAT')
                    for p in r:
                        parts = p.split("=")
                        if parts[0] == "Ub__V" and "u" in self.relevant_for_listeners.lower() or parts[0] == "sw_ver" or parts[0] == "Ib__A" and "i" in self.relevant_for_listeners.lower():
                            try:
                                d[parts[0]] = parts[1]
                                # maybe create method
                                if parts[0] == "Ub__V":
                                    pth = self.generate_path("voltage")
                                    for key, mod in self.downstream_modules_dict.items():
                                        mod.set_upstream_info(self.get_name(), "voltage_path", pth)
                                    date = datetime.datetime.now().strftime('%Y-%m-%d')
                                    fd = open(pth + "/" + date + ".csv", "a+")
                                    header = "ts\tUb__V"
                                    if not os.path.exists(pth + "/" + date + ".csv") or os.path.getsize(pth + "/" + date + ".csv") == 0:
                                        fd.write(header + "\n")
                                    tm = datetime.datetime.now().strftime('%Y-%m-%d-%H-%M-%S')
                                    to_write = tm + "\t" + str(parts[1])
                                    fd.write(to_write + "\n")
                                    fd.close()
                                elif parts[0] == "Ib__A":
                                    pth = self.generate_path("amperage")
                                    for key, mod in self.downstream_modules_dict.items():
                                        mod.set_upstream_info(self.get_name(), "amperage_path", pth)
                                    date = datetime.datetime.now().strftime('%Y-%m-%d')
                                    fd = open(pth + "/" + date + ".csv", "a+")
                                    header = "ts\tIb__A"
                                    if not os.path.exists(pth + "/" + date + ".csv") or os.path.getsize(pth + "/" + date + ".csv") == 0:
                                        fd.write(header + "\n")
                                    tm = datetime.datetime.now().strftime('%Y-%m-%d-%H-%M-%S')
                                    to_write = tm + "\t" + str(parts[1])
                                    fd.write(to_write + "\n")
                                    fd.close()
                                # -------------------
                            except:
                                self.logger.exception('Can\'t assign to dict:')
                    for key, mod in self.downstream_modules_dict.items():
                        try:
                            mod.set_upstream_info(self.get_name(), "hist_firmware_version", d['sw_ver'])
                            # mod.set_hist_fw(d['sw_ver'])
                        except:
                            self.logger.exception('Failed to set sw_ver:')
                        try:
                            mod.set_upstream_info(self.get_name(), "siwim_system_voltage", d['Ub__V'])
                            # mod.set_voltage(d['Ub__V'])
                        except:
                            self.logger.exception('Failed to set V:')

                    msg = self.dict2confstr('mydevice', d)
                    sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
                    try:
                        sock.sendto(msg.encode(), (self.udp_ip, self.udp_port))
                    finally:
                        sock.close()
                    self.zzzzz(self.interval)
            finally:
                ser.close()
        except serial.SerialException as e:
            self.logger.warning(e)
        except:
            self.logger.exception('Fatal error:')
        self.zzzzz(2)
        self.alive = False
