import configparser
import copy
import datetime
import math
import threading
import time
from collections import deque

from lxml import etree

from generic_module import Module


class AggregationModule(Module):
    def __init__(self, args):
        expected_params = {"matching_timeout": 0.0, "fuzziness": 0.0, "boss_receive": None, "root_path": None, "reclassify_by_ff_start": "00:00", "reclassify_by_ff_end": "00:00", "mp_mode": 0}
        Module.__init__(self, args, expected_params)
        self.vehicles_dict = dict()
        self.deque_lock = threading.Lock()
        self.offsets_from_boss_dict = dict()
        self.swd_version = ""
        self.modules_per_lane = dict()
        self.vehicle_classes = None

    def set_swd_version(self, version):
        self.swd_version = version

    def tell_offset(self, recv_name, offsets_dict):
        self.offsets_from_boss_dict[recv_name] = offsets_dict
        self.logger.info('offsets_from_boss_dict: {0}'.format(str(self.offsets_from_boss_dict)))

    def add_vehicle(self, vehicle, recv_module_name):
        sub_vehicle = vehicle.getchildren()[0]
        # in case of no lane data, lane = 0 (no actual lane will ever have the index 0)
        lane = "0"
        try:
            lane = sub_vehicle.find("lane").text
        except:
            pass
            # this happens; no point in logging every occurance
            # self.logger.exception('Failed to obtain lane:')
        if lane not in self.modules_per_lane:
            self.modules_per_lane[lane] = set()
        self.modules_per_lane[lane].add(recv_module_name)
        # print etree.tostring(sub_vehicle)
        with self.deque_lock:
            if recv_module_name not in self.vehicles_dict:
                self.vehicles_dict[recv_module_name] = dict()
            if lane not in self.vehicles_dict[recv_module_name]:
                self.vehicles_dict[recv_module_name][lane] = deque()
            self.vehicles_dict[recv_module_name][lane].append(sub_vehicle)

    def clear_expired(self):
        # this used to clear old external data, but now we want to save it
        # for this reason, we're introducing a new container
        old_ext_data_list = list()
        with self.deque_lock:
            keys = self.vehicles_dict.keys()
            for key in keys:
                for lane in self.vehicles_dict[key]:
                    while len(self.vehicles_dict[key][lane]) > 0:
                        veh_dt = datetime.datetime.strptime(self.vehicles_dict[key][lane][0].find("ts").text, "%Y-%m-%d-%H-%M-%S-%f")
                        now_dt = datetime.datetime.now()
                        td = now_dt - veh_dt
                        # delete older than 3 minutes
                        # making this depentant on matching timeout was probably a mistake
                        if td.total_seconds() > 180:
                            sub_veh = self.vehicles_dict[key][lane].popleft()
                            if self.mp_mode == 0:
                                tmp = etree.Element("vehicle")
                                tmp.append(sub_veh)
                                old_ext_data_list.append(tmp)
                        else:
                            break
        return old_ext_data_list

    def set_flag(self, vehicle, flag):
        first_child = vehicle.getchildren()[0]
        flags_tag = etree.Element("flags")
        flags_tag.text = "00000000"
        try:
            flags_tag = first_child.find("flags")
        except:
            first_child.append(flags_tag)
        flags = int(flags_tag.text, 16)
        mp_flag = flag
        flags |= mp_flag
        flags_tag.text = '{:08x}'.format(flags)

    # --- "private"
    def find_and_form(self, boss_receive_sub_veh):
        ts = boss_receive_sub_veh.find("ts").text
        start = time.mktime(datetime.datetime.strptime(ts, "%Y-%m-%d-%H-%M-%S-%f").timetuple())
        # ---
        vehicle = etree.Element("vehicle")
        vehicle.append(boss_receive_sub_veh)
        lane = "0"
        try:
            lane = boss_receive_sub_veh.find("lane").text
        except:
            pass

        found_list = list()
        found_list.append(self.boss_receive)
        # ---
        # if both boss receive and guest receive event are on the same lane, use that
        # if either boss, or guest (don't), don't have lane defined, use lane "0"
        while True:
            # if we have found everything for this lane, break
            if len(found_list) == len(self.modules_per_lane[lane]):
                break
            for key, value in self.vehicles_dict.items():
                if key not in found_list:
                    try:
                        with self.deque_lock:
                            if lane in value:
                                self.logger.debug(str(key) + " " + str(lane) + " " + str(len(value[lane])))
                                for i in range(len(value[lane])):
                                    self.logger.debug(str(value[lane]))
                                    if self.compare_timestamps(value[lane][i], boss_receive_sub_veh, key) == 1:
                                        if self.mp_mode == 0:
                                            vehicle.append(value[lane][i])
                                        else:
                                            # mp signals flag
                                            # self.set_flag(vehicle, 0x00000001)
                                            self.set_flag(vehicle, 0x00000002)
                                            self.set_flag(vehicle, 0x80000000)
                                        found_list.append(key)
                                        del value[lane][i]
                                        break
                            elif lane != "0" and "0" in value and lane not in value:
                                for i in range(len(value["0"])):
                                    if self.compare_timestamps(value["0"][i], boss_receive_sub_veh, key) == 1:
                                        if self.mp_mode == 0:
                                            vehicle.append(value["0"][i])
                                        else:
                                            # mp signals flag
                                            # self.set_flag(vehicle, 0x00000001)
                                            self.set_flag(vehicle, 0x00000002)
                                            self.set_flag(vehicle, 0x80000000)
                                        found_list.append(key)
                                        del value["0"][i]
                                        # broken = True
                                        break
                    except:
                        self.logger.exception('Find and form error 2:')
                        try:
                            self.deque_lock.release()
                        except:
                            pass
                        break
            if time.time() - start > self.matching_timeout:
                break
            self.zzzzz(0.5)
        return vehicle

        # module_name1 is the name of the "guest" module

    def compare_timestamps(self, sub_veh1, sub_veh2, module_name1):
        try:
            ts1 = sub_veh1.find("ts").text
            ts2 = sub_veh2.find("ts").text
            v = 0
            try:
                v = float(sub_veh2.find("v").text)
            except:
                try:
                    v = float(sub_veh1.find("v").text)
                except:
                    pass
            # determine mp (measuring point); 1 being default
            mp = 1
            try:
                tmp = sub_veh1.find("admpsec").text
                mp = int(tmp[-1])
            except:
                pass
            l1 = sub_veh1.find("lane")
            add_to_offset = sub_veh1.find("siwim_i_add_to_offset")
            parts1 = ts1.split("-")
            parts2 = ts2.split("-")
            t_offset = 0
            try:
                t_offset = self.offsets_from_boss_dict[module_name1][l1.text][mp] / v
                try:
                    factor = self.offsets_from_boss_dict[module_name1][l1.text][mp] / abs(self.offsets_from_boss_dict[module_name1][l1.text][mp])
                    offoffset = float(add_to_offset.text)
                    t_offset = t_offset + (offoffset * factor)
                    if offoffset == 0:
                        offoffset = (float(sub_veh2.find("whlbse").text) + float(sub_veh2.find("append").text)) / v
                        t_offset = t_offset + offoffset
                except:
                    self.logger.debug("offset:", exc_info=True)
            except KeyError as e:
                self.logger.debug(str(e))
            except:
                self.logger.exception('Unexpected error occurred when reading data from offsets_from_boss_dict')
            timestamp1 = time.mktime(datetime.datetime.strptime(ts1, "%Y-%m-%d-%H-%M-%S-%f").timetuple()) + float(parts1[len(parts1) - 1]) / 1000
            timestamp2 = time.mktime(datetime.datetime.strptime(ts2, "%Y-%m-%d-%H-%M-%S-%f").timetuple()) + float(parts2[len(parts2) - 1]) / 1000 + t_offset
            if math.fabs(timestamp1 - timestamp2) <= self.fuzziness:
                self.logger.debug('matched ' + ts1 + ' with ' + ts2)
                self.logger.debug('fuzz: ' + str(self.fuzziness))
                self.logger.debug(str(t_offset))
                return 1
            else:
                if timestamp1 > timestamp2:
                    return 2
                else:
                    # my claim is that we should only clear timestamps (return 3) that are older than self.matching_timeout
                    # twice that should suffice ... possibly ... maybe ...
                    if math.fabs(timestamp1 - timestamp2) > (2 * self.matching_timeout):
                        return 3
                    else:
                        return 4
        except:
            self.logger.exception('Compare timestamps error:')
            return 3

    def helper_safe_vclsses_get(self, subclass, field, typ):
        try:
            return typ(self.vehicle_classes[str(subclass)][field])
        except:
            return None

    def helper_return_element(self, element_name, element_value):
        if element_value == None:
            return None
        element = etree.Element(element_name)
        element.text = str(element_value)
        return element

    def reclassify_wim(self, vehicle, new_cls):
        og_vehicle = copy.deepcopy(vehicle)
        try:
            # an unnecessary check but let's to it anyway
            if self.vehicle_classes == None:
                return vehicle
            ## ready all the relevant classification data
            gvw_limit = self.helper_safe_vclsses_get("subclass_" + str(new_cls), "max_GVW__kN", float)
            axle_limits_str = self.helper_safe_vclsses_get("subclass_" + str(new_cls), "max_axle_weight__kN", str)
            # TODO: try-except?
            axle_limits = dict()
            if axle_limits_str != None:
                for axle_limit in axle_limits_str.split(";"):
                    axle_limit_parts = axle_limit.split(",")
                    axle_limits[axle_limit_parts[0]] = float(axle_limit_parts[1])
            tyre_types_str = self.helper_safe_vclsses_get("subclass_" + str(new_cls), "tyre_type", str)
            tyre_types = list()
            if tyre_types_str != None:
                for tyre_type in tyre_types_str.split(","):
                    tyre_types.append(tyre_type)
            # leave this two as strings
            new_append = self.helper_safe_vclsses_get("subclass_" + str(new_cls), "append__m", str)
            new_prepend = self.helper_safe_vclsses_get("subclass_" + str(new_cls), "prepend__m", str)
            new_cat = self.helper_safe_vclsses_get("subclass_" + str(new_cls), "category", str)
            new_axconfig = self.helper_safe_vclsses_get("subclass_" + str(new_cls), "axle_configuration", str)
            ## reclassify vehicle
            wim = vehicle.find("wim")
            flags = wim.find("flags").text
            # clear everything that needs to be replaced
            # TODO: perhaps make a method for this
            to_clear = ["cls", "axconfig", "cat", "opapp", "append", "prepend", "aol", "vol", "flags", "tyretype", "altclss"]
            for tag in to_clear:
                try:
                    wim.remove(wim.find(tag))
                except:
                    pass
            # replace deleted elements
            new_els = list()
            new_els.append(self.helper_return_element("cls", new_cls))
            new_els.append(self.helper_return_element("cat", new_cat))
            new_els.append(self.helper_return_element("opapp", "I"))
            new_els.append(self.helper_return_element("append", new_append))
            new_els.append(self.helper_return_element("prepend", new_prepend))
            new_els.append(self.helper_return_element("axconfig", new_axconfig))
            aol_el = etree.Element("aol")
            # determine aol
            weight_elements = wim.find("acws").findall("w")
            flags = int(flags, 16)
            # siwim-style flags
            reclass_flag = 0x00000200
            ax_over_flag = 0x00400000
            gvw_over_flag = 0x00800000
            flags |= reclass_flag
            for grp in axle_limits.keys():
                weight_sum = 0
                if "+" in grp:
                    grp_parts = grp.split("+")
                    for grp_part in grp_parts:
                        weight_sum += float(weight_elements[int(grp_part) - 1].text)
                else:
                    weight_sum = float(weight_elements[int(grp) - 1].text)
                if weight_sum > axle_limits[grp]:
                    axgrp_el = etree.Element("axgrp")
                    list_el = etree.Element("list")
                    list_el.text = grp
                    max_el = etree.Element("max")
                    max_el.text = str(weight_sum)
                    over_el = etree.Element("over")
                    over_el.text = str(weight_sum - axle_limits[grp])
                    axgrp_el.append(list_el)
                    axgrp_el.append(max_el)
                    axgrp_el.append(over_el)
                    aol_el.append(axgrp_el)
            if len(aol_el.getchildren()) > 0:
                new_els.append(aol_el)
                flags |= ax_over_flag
            else:
                flags &= ~ax_over_flag
            # determine vol
            gvw = float(wim.find("gvw").text)
            vol_el = etree.Element("vol")
            if gvw_limit != None and gvw > gvw_limit:
                max_el = etree.Element("max")
                max_el.text = str(gvw_limit)
                over_el = etree.Element("over")
                over_el.text = str(gvw - gvw_limit)
            if len(vol_el.getchildren()) > 0:
                new_els.append(vol_el)
                flags |= gvw_over_flag
            else:
                flags &= ~gvw_over_flag
            tyretype_el = etree.Element("tyretype")
            for tyre_type in tyre_types:
                typ = etree.Element("type")
                typ.text = str(tyre_type)
                tyretype_el.append(typ)
            new_els.append(tyretype_el)
            new_els.append(self.helper_return_element("flags", '{:08x}'.format(flags)))
            for new_el in new_els:
                if new_el != None:
                    wim.append(new_el)
        except:
            # return original vehicle
            return og_vehicle
        return vehicle

    def bus_by_ffgroup(self, vehicle):
        wim = vehicle.find("wim")
        anpr = vehicle.find("anpr")
        if wim == None or anpr == None:
            return vehicle
        wim_axconfig = wim.find("axconfig")
        anpr_axconfig_tiein = anpr.find("axconfig_tiein")
        if wim_axconfig == None or wim_axconfig.text == "" or wim_axconfig.text == None or anpr_axconfig_tiein == None:
            return vehicle
        wim_veh_type = wim_axconfig.text[0]
        anpr_veh_type = anpr_axconfig_tiein.text
        # here, we're only dealing with buses; if neither of the two classifications correspond to a bus, there's nothing to do
        if wim_veh_type != "B" and anpr_veh_type != "B" or wim_veh_type == anpr_veh_type:
            return vehicle
        wim_alternatives = list()
        try:
            tmp = wim.find("altclss").getchildren()
            for t in tmp:
                wim_alternatives.append(int(t.text))
        except:
            return vehicle
        wim_cls = int(wim.find("cls").text)
        # this can happen due to siwim-e's reclassify up/down method
        if wim_cls in wim_alternatives:
            wim_alternatives.remove(wim_cls)
        # read classes here to determine inputs for reclassify method (to make it as general as possible)
        clss_pth = self.sites_path + "/" + self.sname + "/conf/vehicle_classes.conf"
        if self.vehicle_classes == None:
            try:
                self.vehicle_classes = configparser.ConfigParser()
                self.vehicle_classes.read(clss_pth)
            except:
                # no reclassifications possible w/o vehicle_classes.conf
                return vehicle
        new_cls = None
        for wim_alternative in wim_alternatives:
            if "subclass_" + str(wim_alternative) in self.vehicle_classes:
                # anpr (ffgroup) veh type is bus, so the vehicle must be a bus
                if anpr_veh_type == "B" and self.vehicle_classes["subclass_" + str(wim_alternative)]["axle_configuration"][0] == "B":
                    new_cls = wim_alternative
                    break
                # anpr (ffgroup) veh type isn't a bus, so the vehicle must be something else
                elif anpr_veh_type != "B" and self.vehicle_classes["subclass_" + str(wim_alternative)]["axle_configuration"][0] != "B":
                    new_cls = wim_alternative
                    break
        if new_cls == None:
            return vehicle
        return self.reclassify_wim(vehicle, new_cls)

    def run(self):
        import time
        self.alive = True
        self.end = False
        self.logger.debug("Starting " + self.name + " aggregation module ...")
        try:
            while True:
                if self.end:
                    self.alive = False
                    self.logger.debug('Thread closed correctly.')
                    return
                    # 8 lanes fapp
                for i in range(9):
                    lane = str(i)
                    if self.boss_receive in self.vehicles_dict and lane in self.vehicles_dict[self.boss_receive] and len(self.vehicles_dict[self.boss_receive][lane]) > 0:
                        boss_receive_sub_veh = self.vehicles_dict[self.boss_receive][lane].popleft()

                        vehicle = self.find_and_form(boss_receive_sub_veh)
                        # reclassify vehicle based on other (ffgroup) info
                        # TODO: perhaps make a whole new module for this (to make it more general)
                        try:
                            if (datetime.datetime.now().hour >= int(self.reclassify_by_ff_start.split(":")[0]) and datetime.datetime.now().hour < int(self.reclassify_by_ff_end.split(":")[0])):
                                vehicle = self.bus_by_ffgroup(vehicle)
                        except:
                            pass
                        # legacy sanity checking
                        if vehicle.find("alpr") != None or vehicle.find("adr") != None:
                            anpr_node = etree.Element("anpr")
                            if vehicle.find("alpr") != None:
                                anpr_node.append(vehicle.find("alpr").find("ts"))
                                anpr_node.append(vehicle.find("alpr").find("lp"))
                                vehicle.find("alpr").getparent().remove(vehicle.find("alpr"))
                            if vehicle.find("adr") != None:
                                anpr_node.append(vehicle.find("adr").find("hgp"))
                                vehicle.find("adr").getparent().remove(vehicle.find("adr"))
                            vehicle.append(anpr_node)

                        for key, mod in self.downstream_modules_dict.items():
                            mod.add_vehicle(copy.deepcopy(vehicle), self.name)

                expired_vehicles = self.clear_expired()
                for vehicle in expired_vehicles:
                    for key, mod in self.downstream_modules_dict.items():
                        mod.add_vehicle(copy.deepcopy(vehicle), self.name)

                time.sleep(0.5)
        except:
            self.logger.exception('Fatal exception in Aggregation Module:')
            self.alive = False
