import base64
import datetime
import io
import json
import math
import os
import platform
import re
import time
from collections import deque

import future.moves.urllib.request as urllib
import piexif
import requests
from PIL import Image
from lxml import html as Html
from piexif import helper
from requests.auth import HTTPDigestAuth

from generic_module import Module


def find_first_match(pattern, source):
    try:
        match = next(re.finditer(pattern, source))
        return source[match.start():match.end()]
    except:
        return None


class CameraModule(Module):

    def who_am_i(self):
        try:
            type_location = "/control/"
            try:
                request = urllib.urlopen(self.protocol + self.host + type_location)
            except Exception as e:  # TODO Remove once it's Python3, and uncomment URLError blow.
                self.logger.warning(e)
                return 'ERR'
            html = request.read()
            if b"MOBOTIX M10" in html:
                return "M10"
            elif b"MOBOTIX M12" in html:
                return "M12"
            elif b"MOBOTIX M15" in html:
                return "M15"
            elif b"MOBOTIX M16" in html:
                return "M16"
            return "ERR"
        # except URLError:  # Python3 only
        #     self.logger.debug('Failed to identify mobotix camera because it did not respond.')
        except:
            self.logger.exception("Mobotix model lookup failed: ")
            return "ERR"

    def __init__(self, args):
        expected_params = {"init_type": None, "host": None, "port": 80, "ignore_classes": {}, "fixed_offset": {}, "cam_focus": {}, "lanes": [], "story": 0, "num_events_to_parse": 10, "max_time_diff": 30, "max_wait_time": 10, "uname": "", "passw": "",
                           "picture_type": "overview"}
        Module.__init__(self, args, expected_params)
        self.logger.debug('Camera type {0}'.format(self.init_type))
        self.protocol = "http://"
        self.definitive_type = self.init_type
        self.vehicles_dict = dict()
        self.missed_count = 0
        self.location = "front"  # may change when first vehicle events arrive; to be implemented

    def generate_path(self, dt, mod):
        pth = self.sites_path
        pth = pth + "/" + self.sname + "/ext"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + mod
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + self.init_type
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + self.name
        if not os.path.exists(pth):
            os.mkdir(pth)
        dt_parts = dt.split("-")
        pth = pth + "/" + dt_parts[0] + "-" + dt_parts[1] + "-" + dt_parts[2] + "-hh-mm-ss"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + dt_parts[0] + "-" + dt_parts[1] + "-" + dt_parts[2] + "-" + dt_parts[3] + "-mm-ss"
        if not os.path.exists(pth):
            os.mkdir(pth)
        # print pth
        return pth

    def add_vehicle(self, vehicle, recv_module_name):
        # sub_vehicle = vehicle.getchildren()[0]
        if recv_module_name not in self.vehicles_dict:
            self.vehicles_dict[recv_module_name] = deque()
        self.vehicles_dict[recv_module_name].append(vehicle)

    def datetime_to_timestamp(self, dtm):
        import datetime
        dt_object = datetime.datetime.strptime(dtm, "%Y-%m-%d-%H-%M-%S-%f")
        parts = dtm.split("-")
        ms = float(parts[len(parts) - 1])
        return time.mktime(dt_object.timetuple()) + ms / 1000.0

    def timestamp_to_datetime(self, ts):
        return datetime.datetime.fromtimestamp(ts).strftime("%Y-%m-%d-%H-%M-%S-%f")

    def crawl(self, root, storage):
        for child in root.getchildren():
            self.crawl(child, storage)
        if len(root.getchildren()) == 0:
            storage.append(root.text)

    def download_and_save(self, cam_ev_list, dtm, ts, lane, auth):
        mindif = self.max_time_diff + 1
        mini = 0
        for i in range(len(cam_ev_list)):
            dif = math.fabs(ts - cam_ev_list[i][1])
            if dif < mindif:
                mini = i
                mindif = dif
                """
                img_ts = cam_ev_list[i][1]
                img_ts_str = str(img_ts)
                ms = img_ts_str[img_ts_str.find(".") + 1:len(img_ts_str)]
                while len(ms) < 3:
                  ms = ms + "0"
                mint = datetime.datetime.fromtimestamp(img_ts).strftime("%Y-%m-%d-%H-%M-%S") + "-" + ms
                """
        if mindif > self.max_time_diff:
            self.logger.warning('Failed to match image within {} seconds.'.format(self.max_time_diff))
            return 1
        cnt = 0
        for i in range(mini - self.story, mini + self.story + 1):
            if i >= 0 and i < len(cam_ev_list):
                req_string = self.protocol + self.host + cam_ev_list[i][2]
                self.logger.debug(req_string)
                try:
                    if not auth:
                        res = requests.get(req_string, timeout=10)
                    else:
                        res = requests.get(req_string, auth=HTTPDigestAuth(self.uname, self.passw), timeout=10)
                    self.logger.debug('Status code for request above: {}.'.format(res.status_code))
                except requests.exceptions.ReadTimeout:
                    self.logger.warning('Timeout while getting photo with {}'.format(req_string))
                    return 1
                if not res.ok:
                    self.logger.warning('Request failed with error {}.'.format(res.status_code))
                    self.logger.debug('Message: {}.\nOriginal request: {}'.format(res.text, req_string))
                    return 1

                result = res.content

                calc_id = i - mini
                str_id = str(abs(calc_id))
                if len(str_id) < 2:
                    str_id = "0" + str_id
                if calc_id < 0:
                    str_id = "n" + str_id
                elif calc_id >= 0:
                    str_id = "p" + str_id
                appendice = str_id

                img_ts = cam_ev_list[i][1]
                img_ts_str = str(img_ts)
                ms = img_ts_str[img_ts_str.find(".") + 1:len(img_ts_str)]
                while len(ms) < 3:
                    ms = ms + "0"
                our_ts = datetime.datetime.fromtimestamp(img_ts).strftime("%Y-%m-%d-%H-%M-%S") + "-" + ms
                # if mint != "":
                user_comment = helper.UserComment.dump("SiWIM_PHOTO_TS:" + our_ts)

                temp_buff = io.BytesIO()
                self.logger.debug('Result is of type {0}'.format(type(result)))
                temp_buff.write(result)
                temp_buff.seek(0)

                try:
                    im = Image.open(temp_buff)
                except:
                    self.logger.warning('Failed to parse image. Response contained: {}'.format(res.text))
                    return 1

                camera = find_first_match(b'CAM=\S+', result)
                exif_dict = {
                    "0th": {
                        piexif.ImageIFD.Artist: camera.replace(b'=', b':') if camera is not None else bytes()
                    },
                    "Exif": {
                        piexif.ExifIFD.UserComment: user_comment
                    }
                }

                exif_bytes = piexif.dump(exif_dict)

                pth = self.generate_path(dtm, "photo")

                try:
                    im.save(pth + "/" + dtm + "_" + lane + "_" + appendice + ".jpg", "jpeg", exif=exif_bytes, quality="keep", optimize=True)
                except:
                    self.logger.warning('Failed to save as image. Saving binary blob: {}/{}_{}.blob'.format(pth, dtm, lane))
                    with open('{}/{}_{}.blob'.format(pth, dtm, lane), 'wb') as blob:
                        blob.write(exif_bytes)
                    return 1
            cnt += 1

        return 0

    def wait_for_future(self, ts):
        now = time.time()
        cnt = 0
        while (now - ts - 3) < 0:
            time.sleep(1)
            now = time.time()
            cnt += 1
            if cnt >= self.max_wait_time:
                if cnt > self.max_wait_time:
                    self.logger.info('Somehow count is higher than max_wait_time, which could lead to an infinite loop in the past ... {} > {}'.format(cnt, self.max_wait_time))
                break

    def apply_offsets(self, ts, lane, v, mp):
        if lane in self.fixed_offset:
            ts = ts + self.fixed_offset[lane] / 1000
        if lane in self.cam_focus and mp in self.cam_focus[lane]:
            ts = ts + self.cam_focus[lane][mp] / v
        return ts

    # versions 10 and 12
    def get_mobotix_1(self, veh):
        self.vehicles_dict
        event_list_loc = "/control/player?eventlist"
        event_info_loc = "/control/eventplayer?get_image_info="
        sub_veh = veh.getchildren()[0]
        lane = int(sub_veh.find("lane").text)
        # determine mp (measuring point); 1 being default
        mp = 1
        try:
            tmp = sub_veh.find("admpsec").text
            mp = int(tmp[-1])
        except:
            pass
        # get picture only for defined lanes
        if lane not in self.lanes:
            return 1
        dtm = sub_veh.find("ts").text
        v = float(sub_veh.find("v").text)
        cls = int(sub_veh.find("cls").text)
        # if class isn't interesting, return NOTHING
        if lane in self.ignore_classes and cls in self.ignore_classes[lane]:
            return 1
            # convert event time to timestamp (seconds from epoch)
        ts = self.datetime_to_timestamp(dtm)
        # offsets
        ts = self.apply_offsets(ts, lane, v, mp)
        dtm_corr = datetime.datetime.fromtimestamp(ts).strftime("%Y-%m-%d-%H-%M-%S")
        # checking if ts is in the future and waiting accordingly
        self.wait_for_future(ts)
        evurl = self.protocol + self.host + event_list_loc
        evurl += "&length=" + str(self.num_events_to_parse)
        evurl += "&s_year=" + dtm_corr.split("-")[0]
        evurl += "&s_month=" + dtm_corr.split("-")[1]
        evurl += "&s_day=" + dtm_corr.split("-")[2]
        evurl += "&s_hour=" + dtm_corr.split("-")[3]
        evurl += "&s_min=" + dtm_corr.split("-")[4]
        evurl += "&s_sec=" + dtm_corr.split("-")[5]
        evurl += "&search=Go"
        request = urllib.urlopen(evurl)
        html = request.read()
        html_obj = Html.fromstring(html)
        events = html_obj.xpath('//a[@title="Show event image"]')
        # find events
        cam_ev_list = list()
        for event in events:
            storage = list()
            self.crawl(event.getparent().getparent(), storage)
            nm = storage[0]
            if len(storage) < 3:
                continue
            dt = storage[1]
            tm = storage[2]
            dttm = dt + " " + tm
            pth = event.attrib["href"]
            cam_ev_list.append([nm, dttm, pth])
            # set accurate timestamp
        for cam_ev in cam_ev_list:
            evurl = self.protocol + self.host + event_info_loc + cam_ev[2]
            # print evurl
            request = urllib.urlopen(evurl)
            html = request.read()
            dtm_id = html.find(cam_ev[1].encode())
            acc_dtm = html[dtm_id:dtm_id + len(cam_ev[1]) + 4]
            facc_dtm = datetime.datetime.strptime(acc_dtm.decode(), "%Y-%m-%d %H:%M:%S.%f").strftime("%Y-%m-%d-%H-%M-%S-%f")
            facc_dtm = facc_dtm[:-3]
            cam_ev[1] = self.datetime_to_timestamp(facc_dtm)
            cam_ev[1] = float(cam_ev[1])
        self.download_and_save(cam_ev_list, dtm, ts, str(lane), False)
        return 0

        # versions 15 and 16

    def get_mobotix_2(self, veh):
        photos_per_sec = 5
        event_list_loc = "/control/event.jpg?output=alarmlist"
        event_img_loc = "/record/events/"
        sub_veh = veh.getchildren()[0]
        lane = int(sub_veh.find("lane").text)
        # determine mp (measuring point); 1 being default
        mp = 1
        try:
            tmp = sub_veh.find("admpsec").text
            mp = int(tmp[-1])
        except:
            self.logger.debug('admpec data missing')
        # get picture only for defined lanes
        if lane not in self.lanes:
            self.logger.debug('Image request failed because {} is not in {}'.format(lane, self.lanes))
            return 2
        dtm = sub_veh.find("ts").text
        self.logger.debug(dtm + "'s turn.")
        v = float(sub_veh.find("v").text)
        cls = int(sub_veh.find("cls").text)
        # if class isn't interesting, return NOTHING
        if lane in self.ignore_classes and cls in self.ignore_classes[lane]:
            self.logger.warning('Image request failed due to vehicle in ignored_classes.')
            return 1
            # convert event time to timestamp (seconds from epoch)
        ts = self.datetime_to_timestamp(dtm)
        # offsets
        ts = self.apply_offsets(ts, lane, v, mp)
        # checking if ts is in the future and waiting accordingly
        self.wait_for_future(ts)
        # getting event list
        tsofs = self.num_events_to_parse / (2 * photos_per_sec)
        evurl = self.protocol + self.host + event_list_loc
        evurl += "&length=" + str(self.num_events_to_parse)
        evurl += "&searchbytime_start=" + str(ts + tsofs)  # TODO; do I need to add bias?
        try:
            request = urllib.urlopen(evurl, timeout=5)
        except:  # URLError when this is Python3 only
            self.logger.warning('Image request failed due to timeout.')
            return 1
        if platform.python_version_tuple()[0] == '2':
            try:
                if request.code != 200:
                    self.logger.warning('Image request failed with status {}. Message {}'.format(request.code, request.reason))
                    return 1
            except Exception as e:
                self.logger.error(e)
        else:
            if request.status != 200:
                self.logger.warning('Image request failed with status {}. Message {}'.format(request.status, request.reason))
                return 1

        # json array of events
        html = request.read()
        json_array = json.loads(html)
        cam_ev_list = list()
        for jso in json_array:
            aiv = jso["alarmimage"]
            tsv = float(jso["timestamp"])
            if len(aiv) != 8:
                continue
            pth = event_img_loc + aiv[0:3] + "/" + aiv[3:6] + "/E00000.jpg"
            cam_ev_list.append((aiv, tsv, pth))
        self.logger.debug('Received that many events: ' + str(len(cam_ev_list)))
        if len(cam_ev_list) == 0:
            self.logger.warning('{} response: {}'.format(evurl, html.decode()))
            return 1
        if self.download_and_save(cam_ev_list, dtm, ts, str(lane), False) == 1:
            self.logger.warning('Downloading failed!')
            return 1
        return 0

    def get_axis(self, veh):
        event_list_loc = self.protocol + self.host + ":" + str(self.port) + "/local/image_retrieve/list.cgi?"  # add number of events & ts
        sub_veh = veh.getchildren()[0]
        lane = int(sub_veh.find("lane").text)
        # determine mp (measuring point); 1 being default
        mp = 1
        try:
            tmp = sub_veh.find("admpsec").text
            mp = int(tmp[-1])
        except:
            pass
        # get picture only for defined lanes
        if lane not in self.lanes:
            self.logger.debug('Lane {} is ignored.'.format(lane))
            return 2
        dtm = sub_veh.find("ts").text
        v = float(sub_veh.find("v").text)
        clss = int(sub_veh.find("cls").text)
        # if class isn't interesting, return NOTHING
        if lane in self.ignore_classes and clss in self.ignore_classes[lane]:
            self.logger.debug('Class {} is ignored.'.format(clss))
            return 1
        # convert event time to timestamp (seconds from epoch)
        ts = self.datetime_to_timestamp(dtm)
        # offsets
        ts = self.apply_offsets(ts, lane, v, mp)
        # checking if ts is in the future and waiting accordingly
        self.wait_for_future(ts)
        try:
            res = requests.get(event_list_loc + "?n=" + str(self.num_events_to_parse) + "?ts=" + self.timestamp_to_datetime(ts)[:-3], auth=HTTPDigestAuth(self.uname, self.passw), timeout=10)
        except (requests.exceptions.ConnectionError, requests.exceptions.ReadTimeout):
            self.logger.warning('Connection to {} refused!'.format(event_list_loc + "?n=" + str(self.num_events_to_parse) + "?ts=" + self.timestamp_to_datetime(ts)[:-3]))
            return 1
        # list of picture timestamps
        con = res.content
        lines = str(con.decode()).split("<br/>")
        lines = lines[:-1]
        cam_ev_list = list()
        for line in lines:
            aiv = "rage!"
            tsv = self.datetime_to_timestamp(line)
            pth = ":" + str(self.port) + "/local/image_retrieve/get.cgi?img=" + line + ".jpeg"
            cam_ev_list.append((aiv, tsv, pth))
        self.logger.debug('Received that many events: ' + str(len(cam_ev_list)))
        if len(cam_ev_list) == 0:
            self.logger.debug('Event query response: ' + con.decode())
        self.download_and_save(cam_ev_list, dtm, ts, str(lane), True)
        return 0

    def get_technix(self, veh):
        sub_veh = veh.getchildren()[0]
        anpr_location = "/ANPRArchive/"
        dtm = sub_veh.find("ts").text
        lane = sub_veh.find("lane").text
        ip = "\\"
        path = None
        if self.picture_type == "anpr":
            ip = sub_veh.find("anpr_ip").text
            path = self.generate_path(dtm, "anpr")
        elif self.picture_type == "adr":
            ip = sub_veh.find("adr_ip").text
            path = self.generate_path(dtm, "anpr")
        elif self.picture_type == "overview":
            ip = sub_veh.find("trg_ip").text
            path = self.generate_path(dtm, "photo")
        ip = ip.replace("\\", "/")
        request = urllib.Request(self.protocol + self.host + ":" + str(self.port) + anpr_location + ip)
        base64string = base64.b64encode('%s:%s' % (self.uname, self.passw))
        request.add_header("Authorization", "Basic %s" % base64string)
        result = urllib.urlopen(request).read()
        fd = open(path + "/" + dtm + "_" + lane + "_p00.jpg", "wb")
        fd.write(result)
        fd.close()

    def get_macq(self, veh):
        sub_veh = veh.find("wim")
        if sub_veh == None:
            return
        anpr_veh = veh.find("anpr")
        if anpr_veh == None:
            return
        image_pth = "http://" + self.host + ":" + str(self.port) + "/meci-L1/discr/images/" + anpr_veh.find("trg_ip").text
        dtm = sub_veh.find("ts").text
        lane = sub_veh.find("lane").text
        if self.picture_type == "anpr":
            image_pth = "http://" + self.host + ":" + str(self.port) + "/meci-L1/discr/images/" + anpr_veh.find("anpr_ip").text
        path = self.generate_path(dtm, "photo")
        res = requests.get(image_pth)
        with open('{0}/{1}_{2}_p00.jpg'.format(path, dtm, lane), 'wb') as f:
            for chunk in res.iter_content(1024):
                f.write(chunk)

    def get_cammra(self, veh):  # Not a typo.
        sub_veh = veh.find("wim")
        if sub_veh == None:
            return
        lane = sub_veh.find("lane").text
        if len(self.lanes) > 0 and int(lane) not in self.lanes:
            return
        anpr_vehs = veh.findall("anpr")
        if len(anpr_vehs) == 0:
            return
        anpr_veh = None
        for av in anpr_vehs:
            if av.find("cam_addr").text == self.host:
                anpr_veh = av
                break
        if anpr_veh == None:
            return
        dtm = sub_veh.find("ts").text
        path = ""
        image_pth = ""
        if self.picture_type == "anpr":
            image_pth = "http://" + self.host + ":" + str(self.port) + anpr_veh.find("trg_ip").text
            path = self.generate_path(dtm, "anpr")
        elif self.picture_type == "overview":
            image_pth = "http://" + self.host + ":" + str(self.port) + anpr_veh.find("trg_ip").text
            path = self.generate_path(dtm, "photo")
        try:
            # keep this as default for now (remove in the future)
            u = "root"
            p = "axis"
            if self.uname != "" and self.passw != "":
                u = self.uname
                p = self.passw
            res = requests.get(image_pth, auth=HTTPDigestAuth(u, p))
            # content = res.content
            # see if it works with no "amp;" (testing showed that it might)
            # if content == "":
            # image_pth = image_pth.replace("amp;", "")
            # res = requests.get(image_pth, auth=HTTPDigestAuth(u, p))
            img_name = "{}_{}_p00".format(dtm, lane)
            if res.ok:  # If response is as expected.
                self.logger.debug("Saving image")
                if self.picture_type == "anpr":
                    coords = anpr_veh.find("lp_coordinates").text[1:-1].replace(" ", "").split(",")
                    temp_buff = io.BytesIO()
                    temp_buff.write(res.content)
                    temp_buff.seek(0)
                    try:
                        im = Image.open(temp_buff)
                        width, height = im.size
                        im_c = im.crop((int(coords[0]), int(coords[1]), int(coords[0]) + int(coords[2]), int(coords[1]) + int(coords[3])))
                        # im_c = im.crop((int(coords[0]), int(coords[1]), width, height))
                        # im_c = im.crop((int(coords[0]), int(coords[1]), width - (int(coords[0]) + int(coords[2])), height - (int(coords[1]) + int(coords[3]))))
                        im_c.save("{}/{}.jpg".format(path, img_name), "jpeg", optimize=True)
                    # except UnidentifiedImageError:  # TODO Enable for v5
                    #     self.logger.error('Received image was not in correct format. Saving raw data as {}.dump.'.format(img_name))
                    #     with open('{}/{}.dump'.format(path, img_name), 'wb') as img:
                    #         img.write(res.content)
                    #     return 1
                    except:
                        self.logger.exception("Save/crop exception: ")
                        self.logger.debug('Failed to parse/crop image. Response contained: {}'.format(res.content))
                        return 1
                else:
                    with open('{0}/{1}_{2}_p00.jpg'.format(path, dtm, lane), 'wb') as f:
                        f.write(res.content)
                        self.logger.info("Image saved.")
            elif res.status_code == 404:  # If image is missing.
                self.logger.warning('No photo for {}'.format(img_name))
            else:
                self.logger.error('Camera returned status code: {} and message: {}'.format(res.status_code, res.text))
        except:
            self.logger.exception('General exception:')

    def save_arh(self, veh):
        sub_veh = veh.find("wim")
        if sub_veh == None:
            return
        lane = sub_veh.find("lane").text
        if len(self.lanes) > 0 and int(lane) not in self.lanes:
            return
        anpr_vehs = veh.findall("anpr")
        if len(anpr_vehs) == 0:
            return
        anpr_veh = anpr_vehs[0]
        dtm = sub_veh.find("ts").text
        img_content = ""
        if self.picture_type == "anpr":
            img_content = anpr_veh.find("lprimage").text
        if img_content != "":
            path = self.generate_path(dtm, "anpr")
            image_blob = base64.b64decode(img_content)
            fd = open('{0}/{1}_{2}_p00.jpg'.format(path, dtm, lane), "wb")
            fd.write(image_blob)
            fd.close()

    def run(self):
        self.alive = True
        self.end = False

        if self.init_type == "mobotix":
            self.definitive_type = self.who_am_i()

        for key, mod in self.downstream_modules_dict.items():
            mod.set_upstream_info(self.name, "camera", {"name": self.name, "type": self.definitive_type, "ip": self.host, "perspective": self.picture_type})
        try:
            while True:
                if self.definitive_type == "ERR":
                    self.alive = False
                    self.logger.warning('Error while getting Mobotix model. Exiting.')
                    return
                if self.end == True:
                    self.alive = False
                    self.logger.debug('Thread closed correctly.')
                    return
                elif self.missed_count >= 10:
                    self.missed_count = 0
                    self.alive = False
                    self.logger.warning('Last 10 vehicles had no photo. Closing thread.')
                    return
                for key in self.vehicles_dict:
                    if len(self.vehicles_dict[key]) > 0:
                        try:
                            sub_veh = self.vehicles_dict[key].popleft()
                        except IndexError:
                            self.logger.error('Couldn\'t pop from {} for key {}.'.format(self.vehicles_dict, key))
                        if self.definitive_type == "M10" or self.definitive_type == "M12":
                            try:
                                if self.get_mobotix_1(sub_veh) == 1:
                                    self.missed_count += 1
                                else:
                                    self.missed_count = 0
                            except:
                                self.missed_count += 1
                                self.logger.exception('Failed to get mobotix_1:')
                        elif self.definitive_type == "M15" or self.definitive_type == "M16":
                            try:
                                ret = self.get_mobotix_2(sub_veh)
                                if ret == 1:  # If getting photo failed.
                                    self.missed_count += 1
                                elif ret == 0:  # If getting photo succeeded.
                                    self.missed_count = 0
                            except:
                                self.missed_count += 1
                                self.logger.exception('Failed to get mobotix_2:')
                        elif self.definitive_type == "technix":
                            try:
                                self.get_technix(sub_veh)
                            except:
                                self.logger.exception('Failed to get technix:')
                        elif self.definitive_type == "axis":
                            try:
                                ret = self.get_axis(sub_veh)
                                if ret == 1:  # If getting photo failed.
                                    self.missed_count += 1
                                elif ret == 0:  # If getting photo succeeded.
                                    self.missed_count = 0
                            except:
                                self.missed_count += 1
                                self.logger.exception('Failed to get axis:')
                        elif self.definitive_type == "macq":
                            try:
                                self.get_macq(sub_veh)
                            except:
                                self.logger.exception('Failed to get macq:')
                        elif self.definitive_type == "ffgroup":
                            try:
                                self.get_cammra(sub_veh)
                            except:
                                self.logger.exception('Failed to get ffgroup:')
                        elif self.definitive_type == "arh":
                            try:
                                self.save_arh(sub_veh)
                            except:
                                self.logger.exception('Failed to save arh:')
                        else:
                            self.logger.warning('A non-anticipaded camera type.')
                            self.alive = False
                            return
                self.zzzzz(0.5)
        except:
            self.logger.exception('Fatal error:')
            self.alive = False
