# This file contains various helper functions that are used across the application.
import platform
import sys
from configparser import ConfigParser, MissingSectionHeaderError
from pathlib import Path
from typing import Dict

from cestel_helpers.exceptions import ConfError

import config
from consts import CONF_BACKUP_DIR, CONF_DIR, LOG_DIR


def get_current_site(siwim_e_conf: Path, set_i_dir: bool = True) -> str:
    """ Returns current site or 'generic' if site name can't be read.
    :param siwim_e_conf: Path to siwim.conf file.
    :param set_i_dir: Indicates whether directories for I should be created within the site folder.
    """
    cfg = ConfigParser(strict=False)  # Ignore potential duplicate keys in conf.
    try:
        cfg.read(siwim_e_conf)
    except (FileNotFoundError, MissingSectionHeaderError) as e:
        raise ConfError(e)
    new_site = cfg.get('global', 'site', fallback='generic')
    if new_site != config.site_name:  # If site changed, we have to update conf and logging directories.
        config.conf_backup_dir = Path(config.sites_dir, new_site, CONF_BACKUP_DIR)
        config.conf_backup_dir.mkdir(parents=True, exist_ok=True)
        if set_i_dir:
            config.conf_dir = Path(config.sites_dir, new_site, CONF_DIR)
            config.conf_dir.mkdir(parents=True, exist_ok=True)
            config.log_dir = Path(config.sites_dir, new_site, LOG_DIR)
            config.log_dir.mkdir(parents=True, exist_ok=True)
    return new_site


def get_siwim_e_version(root_path: str):
    if platform.system() == 'Windows':
        def w32get_file_info(filename, info):
            from ctypes import windll, byref, create_string_buffer, c_uint, string_at, c_void_p, wstring_at
            import sys
            import array
            mygetfileversionsize = windll.version.GetFileVersionInfoSizeW if sys.version_info[0] == 3 else windll.version.GetFileVersionInfoSizeA
            mygetfileversioninfo = windll.version.GetFileVersionInfoW if sys.version_info[0] == 3 else windll.version.GetFileVersionInfoA
            myverqueryvalue = windll.version.VerQueryValueW if sys.version_info[0] == 3 else windll.version.VerQueryValueA
            mystringat = wstring_at if sys.version_info[0] == 3 else string_at
            size = mygetfileversionsize(filename, None)
            if size == 0:
                return ''
            res = create_string_buffer(size)
            mygetfileversioninfo(filename, None, size, res)
            r = c_void_p(0)
            l = c_uint(0)
            myverqueryvalue(res, '\\VarFileInfo\\Translation', byref(r), byref(l))
            if not l.value:
                return ''
            codepages = array.array('H', string_at(r.value, l.value))
            codepage = tuple(codepages[:2].tolist())
            myverqueryvalue(res, ('\\StringFileInfo\\%04x%04x\\' + info) % codepage, byref(r), byref(l))
            return mystringat(r.value, l.value - 1)

        return w32get_file_info(str(Path(root_path, 'siwim_e/bin/siwim_mcp.exe')), "FileVersion")
    elif platform.system() == 'Linux':
        # try:
        #     return re.search(r'\d+\.\d+\.\d+', check_output([Path(root_path, 'siwim-e/bin/siwim_mcp'), '-h'], timeout=10).decode()).group()
        # except subprocess.CalledProcessError:
        #     return ''
        return ''


def exit_with_prompt() -> None:
    """ Convenience function for asking user for input and then exit. """
    input('Press ENTER to close.')
    sys.exit(1)


def dict2confstr(section: str, d: Dict[str, str]) -> str:
    ini_string = f'[{section}]\n'
    for k, v in d.items():
        ini_string += f'{k}={v}\n'
    return ini_string
