import platform
import socket
import subprocess
import time

import serial
import serial.tools.list_ports

from consts import HIST_BAUD, HIST_INTERVAL, HIST_IP, HIST_LISTENERS, HIST_PORT, HIST_SERIAL, HIST_VPN, HIST_VPN_TIMEOUT, MOD_HYST, SAVE_HEADER, SAVE_PATH, SAVE_SUFFIX
from exceptions import StopModule
from generic_module import Module
from helpers.helpers import dict2confstr


class HistModule(Module):
    baud: int
    interval: int
    vpn_ping_num: int
    relevant_for_listeners: str
    serial_port: str
    udp_ip: str
    udp_port: int
    vpn_host: str

    def __init__(self, args):
        self.vpn_ping_num: int = 0
        Module.__init__(self, args, mandatory_keys=(HIST_BAUD, HIST_INTERVAL, HIST_IP, HIST_LISTENERS, HIST_PORT, HIST_SERIAL, HIST_VPN), optional_keys=(HIST_VPN_TIMEOUT,))
        self.type = MOD_HYST
        if self.vpn_ping_num > 30:
            self.logger.warning('Value vpn_ping_num has an illogical value. It is recommended to remove it from configuration file.')
            self.vpn_ping_num = 30
        # automatic detection if applicable
        potential_ports = self.get_hist_ports()
        if self.serial_port == "auto":
            if len(potential_ports) == 1:
                self.serial_port = potential_ports[0]
            else:
                self.logger.critical(f'Automatic port detection is enabled, but there is more than one viable port! Please manually pick the correct one: {potential_ports}')
                self.zzzzz(2)
                return

    def is_in_vpn(self):
        try:
            if platform.system() == "Windows":
                ping = subprocess.Popen(["ping", "-n", str(self.vpn_ping_num), "-w", "2", self.vpn_host], stdout=subprocess.PIPE).communicate()[0].decode()
            elif platform.system() == 'Linux':
                ping = subprocess.check_output(['ping', '-c', str(self.vpn_ping_num), '-W', '2', self.vpn_host]).decode()
            else:
                raise NotImplementedError(f'Sending ping is not implemented for {platform.platform()}.')
        except subprocess.CalledProcessError:
            self.logger.warning(f'Pinging {self.vpn_host} failed. System may not be in VPN.')
            return False
        if "ttl=" in ping.lower():
            return True
        return False

    def get_hist_ports(self):
        usb_serial_ports = []
        ports = serial.tools.list_ports.comports()  # This returns a list of ListInfoPort objects for PySerial 3.4 (Win10 machines) and generator of tuples for PySerial 2.7 (WinXP machines)
        if platform.system() == 'Windows':
            for port in ports:
                if 'USB Serial Port' in port.description:  # In PySerial 3.4 ports are stored in a ListInfoPort object.
                    usb_serial_ports.append(port.device)
        elif platform.system() == 'Linux':
            for port in ports:
                if port[1] == 'FT230X Basic UART':
                    usb_serial_ports.append(port[0])
        else:
            raise NotImplementedError(f'Can\'t obtain hist port for {platform.platform()}')
        return usb_serial_ports

    def histcmd(self, ser, cmnd, resp='', to=0.5):
        if ser.timeout != to:
            ser.timeout = to
        ser.flushInput()
        ser.write((cmnd + '\n').encode())
        r = []
        while True:
            l = ser.readline().rstrip().decode()
            if l == '':  # EOF
                return r
            r.append(l)
            if len(resp) > 0 and l.find(resp) != -1 or len(r) > 10:  # resp found or safeguard
                return r

    def run(self):
        self.alive = True
        self.end = False
        try:
            ser = serial.Serial(self.serial_port, self.baud)
            try:
                while True:
                    self.throttle()
                    # if server connection
                    if self.vpn_ping_num > 0 and not self.is_in_vpn():
                        raise StopModule(f'Resetting due to no connection with VPN IP: {self.vpn_host}')
                    if self.end:
                        ser.close()
                        self.alive = False
                        raise StopModule('Thread closed correctly.')
                    d = {'ts': time.strftime('%Y-%m-%d-%H-%M-%S')}
                    r = self.histcmd(ser, 'STAT')
                    for p in r:
                        parts = p.split("=")
                        if parts[0] == "Ub__V" and "u" in self.relevant_for_listeners.lower() or parts[0] == "sw_ver" or parts[0] == "Ib__A" and "i" in self.relevant_for_listeners.lower():
                            d[parts[0]] = parts[1]
                            if parts[0] == "Ub__V":
                                self.write_data({SAVE_PATH: 'voltage', SAVE_HEADER: 'ts\tUb__V', SAVE_SUFFIX: 'tsv'}, parts[1])
                                self.set_upstream_info(self.get_name(), 'voltage_path', self.data_dirs['voltage'])
                            elif parts[0] == "Ib__A":
                                self.write_data({SAVE_PATH: 'amperage', SAVE_HEADER: 'ts\tIb__A', SAVE_SUFFIX: 'tsv'}, parts[1])
                                self.set_upstream_info(self.get_name(), 'amperage_path', self.data_dirs['amperage'])
                    self.set_upstream_info(self.get_name(), "hist_firmware_version", d['sw_ver'])
                    self.set_upstream_info(self.get_name(), "siwim_system_voltage", d['Ub__V'])

                    msg = dict2confstr('mydevice', d)
                    sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
                    try:
                        sock.sendto(msg.encode(), (self.udp_ip, self.udp_port))
                    finally:
                        sock.close()
                    self.zzzzz(self.interval)
            finally:
                ser.close()
        except (serial.SerialException, StopModule) as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
        self.zzzzz(2)
        self.alive = False
