from typing import Dict, Optional

import requests
from requests.auth import HTTPDigestAuth

from acquisition.camera import CameraModule
from consts import CAM_HOST, CAM_LANES, CAM_OFFSET, CAM_PASS, CAM_PIC_TYPE, CAM_PORT, CAM_STORY, CAM_USER, MOD_TYPE, RCV_OFFSET
from exceptions import NoPhoto, StopModule


def get_urls(protocol: str, ip: str, port: int) -> Optional[Dict[str, str]]:
    """ This function gets urls based on the first path that exists on the camera. """
    try:
        if requests.get(f'{protocol}{ip}:{port}/local/imageretrieve/settings.cgi', timeout=3).status_code != 404:  # V12
            return {'list': f'{protocol}{ip}:{port}/local/imageretrieve/settings.cgi?name=list&', 'get': f'{protocol}{ip}:{port}/local/imageretrieve/settings.cgi?name=get&', 'separator': '&'}
        elif requests.get(f'{protocol}{ip}:{port}/local/image_retrieve/list.cgi', timeout=3) != 404:  # V11
            return {'list': f'{protocol}{ip}:{port}/local/image_retrieve/list.cgi?', 'get': f'{protocol}{ip}:{port}/local/image_retrieve/get.cgi?', 'separator': '?'}
    except requests.exceptions.ConnectTimeout:
        return None
    return None


class CameraAxis(CameraModule):
    def __init__(self, args):
        # Initialize the module, passing it a list of keys that must exist for it to work properly.
        CameraModule.__init__(self, args=args, mandatory_keys=(CAM_HOST, CAM_PIC_TYPE, CAM_PORT, CAM_LANES, CAM_STORY, MOD_TYPE, RCV_OFFSET), optional_keys=(CAM_PASS, CAM_OFFSET, CAM_USER))
        self.urls = get_urls(self.protocol, self.host, self.port)  # This contains list and photo urls for the camera type.

        # Determine correct list and get urls.
        if not self.urls:
            raise StopModule(f'No axis camera found at {self.protocol}{self.host}:{self.port}.')
        self.logger.debug(f'Using base urls: {self.urls}.')

    def get_photo(self, veh) -> None:
        event_list_loc = self.urls['list']  # add number of events & ts

        # Extract relevant data from xml.
        try:
            veh_ts, photo_ts, lane = self.get_photo_data(veh)
        except NoPhoto as e:  # This occurs if lane is wrong.
            self.logger.warning(e)
            return

        try:
            event_list_loc += f'n={self.story * 2 + 1}{self.urls["separator"]}ts={self.timestamp_to_datetime(photo_ts)}'
            response = requests.get(event_list_loc, auth=HTTPDigestAuth(self.uname, self.passw), timeout=10)
        except (requests.exceptions.ConnectionError, requests.exceptions.ReadTimeout):
            raise NoPhoto(f'Connection to {event_list_loc} refused!')
        # list of picture timestamps
        con = response.content
        lines = str(con.decode()).split('<br/>')[:-1]
        cam_ev_list = []
        for line in lines:
            try:
                tsv = self.datetime_to_timestamp(line)
            except ValueError:
                raise NoPhoto(f'Response for {event_list_loc} is "{line}", which is not a valid date.')
            pth = self.urls['get'][self.urls['get'].find(':', len(self.protocol)):] + f'img={line}.jpeg'  # TODO Needs refactoring, probably a rewrite of download_and_save to not expect such an arbitrary argument.
            cam_ev_list.append((tsv, pth))
        if len(cam_ev_list) == 0:
            self.logger.debug(f'Event query response: {con.decode()}')
        self.download_and_save(cam_ev_list, veh_ts, photo_ts, lane, True)

    def run(self) -> None:
        self.alive = True
        self.end = False

        try:
            self.main()
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
            self.alive = False
