#!/usr/bin/python3
#version = 1.1.0.0

import json, os, configparser, datetime

class SystemData():

    def __init__(self):
        self.conf_file = configparser.ConfigParser()
        self.conf_file.read("D:\zabbix_scripts\conf.ini")
        self.active_site_name = self.get_site_name()
        self.current_date = datetime.date.today()
        self.results = {'data': list(), 'other': list(), 'triggers': []}
        
        #find all temperatures that need to be alarmed when missfunctioned
        self.results["alarmed_temp"] = self.important_temperature()

        self.read_amp()
        self.read_voltage()
        self.read_temperature()
        self.trigger_check()


    def get_results(self):
        '''
        Returns results
        '''
        return self.results

    def get_site_name(self):
        '''
        Finds the latest site name on the system.
        '''
        conf_path = self.conf_file["conf"]["e_conf"]
        conf_parser = configparser.ConfigParser()
        conf_parser.read(conf_path)
        site_name = conf_parser.get('global', 'site')
        return site_name
    
    def get_number_of_lanes(self):
        '''
        Finds how many lanes are on specific site
        '''
        conf_path = self.conf_file["conf"]["site_conf"]
        conf_parser = configparser.ConfigParser()
        #reads conf
        conf_parser.read(conf_path.format(site_name=self.active_site_name))
        num_of_lanes = int(conf_parser.get('bridge', 'number_of_lanes'))
        return num_of_lanes

    def most_recet_dir(self, path):
        '''
        Finds the latest dir name in given path.
        '''
        #default folder path for sites
        os.chdir(path)

        #gets all the files sorted by date
        files = sorted(filter(os.path.isdir, os.listdir('.')), key=os.path.getctime)
        return files[-1]

    def important_temperature(self):
        '''
        Finds which temperatures are important and will be alarmed if they show strange values
        Ti temperature is always here + the one which is in site.conf file and used fro temperature compensation
        '''
        alarmnd_temp = ["Ti"]
        try:
            conf_path = self.conf_file["conf"]["cf_conf"]
            #check if there is cf file or cf1 and cf2
            if os.path.exists(conf_path.format(site_name=self.active_site_name)):
                conf_parser = configparser.ConfigParser()
                #reads conf
                conf_parser.read(conf_path.format(site_name=self.active_site_name))
                used_temp = conf_parser.get('global', 'use_temperature')
                alarmnd_temp.append(used_temp)
            else:
                for i in range(2):
                    conf_path = self.conf_file["conf"]["cf_conf"].replace("cf.conf", "cf{0}.conf".format(i+1))
                    conf_parser = configparser.ConfigParser()
                    #reads conf
                    conf_parser.read(conf_path.format(site_name=self.active_site_name))
                    used_temp = conf_parser.get('global', 'use_temperature')
                    alarmnd_temp.append(used_temp)

        except:
            self.results["errors"] = "Something wrong with cf.conf reading"
        return alarmnd_temp

    def most_recent_data(self, path, want_first=False):
        '''
        Return first and last last line from a most recent file in given path
        '''
        os.chdir(path)
        files = sorted(filter(os.path.isfile, os.listdir(".")), key=os.path.getmtime)
        #gets the file with last "modified date"
        active_file = files[-1]

        #file name without extension
        file_name = active_file.split(".")[0] 
        if file_name != str(self.current_date):
            return []

        with open(active_file) as file:
            lines = file.readlines()
            if want_first:
                first_line = lines[0].replace("\n", "").split("\t")
            newest_data = lines[-1].replace("\n", "").split("\t")
        
        return [first_line, newest_data] if want_first else newest_data

    def read_amp(self):
        '''
        Reads amperage and adds it to results
        '''
        path = self.conf_file["amp_volt_temp_log"]["amp_path"].format(site_name=self.active_site_name)
        try:
            latest_path = path + '\\' + self.most_recet_dir(path)
            amp_data = self.most_recent_data(latest_path)[-1]
        except:
            amp_data = "CANT FIND FILE"
        self.results['other'].append({'amperage': amp_data})

    def read_voltage(self):
        '''
        Reads voltage and adds it to results
        '''
        path = self.conf_file["amp_volt_temp_log"]["vol_path"].format(site_name=self.active_site_name)
        try:
            latest_path = path + '\\' + self.most_recet_dir(path)
            volt_data = self.most_recent_data(latest_path)[-1]
        except:
            volt_data = "CANT FIND FILE"
        self.results['other'].append({'voltage': volt_data})

    def read_temperature(self):
        '''
        Reads temperature from all folders in /ext/temperature
        '''
        path = self.conf_file["amp_volt_temp_log"]["temp_path"].format(site_name=self.active_site_name)
        try:
            can_path = path + '\\' + "can"
            latest_path = can_path + '\\' + self.most_recet_dir(can_path)
            temp_data = self.most_recent_data(latest_path)
        except:
            temp_data = []
            self.results["errors"] = "cant read can folder"
        for i, one in enumerate(temp_data[1:]):
            if eval(one) > -100:
                #data is different from default so we need to save it
                channel_number = i + 1 if i != len(temp_data[1:]) - 1 else 'i'
                channel_name = "T{0}".format(channel_number)
                self.results['data'].append({'name': channel_name, 'temperature': one})
        
        #papago 
        papago_path = path + '\\' + "papago"
        if os.path.exists(papago_path):
            #there is papago folder add temps
            os.chdir(papago_path)
            all_folders = filter(os.path.isdir, os.listdir("."))
            for folder in all_folders:
                full_path = papago_path + '\\' + folder
                try:
                    first_line, temp_data = self.most_recent_data(full_path, True)
                    for i, one in enumerate(first_line[1:]):
                        channel_name = one.split("__")[0]
                        channel_temp = temp_data[1:][i]
                        if channel_temp != "-9999.00":
                            self.results['data'].append({'name': channel_name, 'temperature': channel_temp})
                    #return back
                    os.chdir(papago_path)
                except:
                    self.results["errors"] = "cant find papago folder"


    def trigger_check(self):
        '''
        Method with check if at least 2 triggers per lane are set up and if more than 2 triggers at least 2 of them have role weigh.
        Will also check if at least one trigger per lane is below 0.05V
        '''
        number_of_lanes = self.get_number_of_lanes()
        tsplit_path = self.conf_file["conf"]["tsplit_conf"].format(site_name=self.active_site_name)
        conf_parser = configparser.ConfigParser()
        section_type = "trigger_above__V"
        #read tsplit values check if there is one tsplit or tspli1 and tsplit2
        if os.path.exists(tsplit_path):
            #Check in tsplit file for trigger values
            conf_parser.read(tsplit_path)
            if section_type in conf_parser:
                all_trigers = conf_parser.items(section_type)

        else:
            #There are probably two tsplit files (tsplit1, tsplit2)
            all_trigers = list()
            path, suffix = tsplit_path.split(".")
            for i in range(2):
                new_path = path + str(i + 1) + "." + suffix
                conf_parser.read(new_path)   
                if section_type in conf_parser:
                    all_trigers += conf_parser.items(section_type)    

        #counting triggers below 0.05V
        below = 0
        #couting weigh role
        weigh_roles = 0
        site_conf = self.conf_file["conf"]["site_conf"]
        conf_parser.read(site_conf.format(site_name=self.active_site_name))
        for ch, value in all_trigers:
            role = conf_parser.get('sensors', "{0}_role".format(ch))
            if "weigh" in role:
                weigh_roles += 1
            if float(value) < 0.05:
                below += 1

        self.results["triggers"].append({"number_of_per_lane": len(all_trigers) / number_of_lanes})
        self.results["triggers"].append({"below_0_05V_per_lane": below / number_of_lanes})
        self.results["triggers"].append({"weigh_role_per_lane": weigh_roles / number_of_lanes})


        
if __name__ == "__main__":
    tracker = SystemData()
    print(json.dumps(tracker.get_results()))