#!/usr/bin/python3
#version = 1.3.2.0

import re, json, psutil, configparser, sys
import xml.etree.ElementTree as ET

class Process_tracker:
    
    def __init__(self, type):
        self.conf_file = configparser.ConfigParser()
        self.conf_file.read("D:\zabbix_scripts\conf.ini")

        self.results = dict()
        if type == "processes":
            self.measured_proc = json.loads(self.conf_file.get("tracking_processes", "track_list"))
            self.ignored_proc = json.loads(self.conf_file.get("tracking_processes", "ignore_list"))
            self.valid_pids = list()

            self.results["other_processes"] = list()
            self.calculate_all_data()

        elif type == "drives":
            self.get_drive_sizes()

        elif type == "sw_hw":
            self.get_sw_hw_data()
        else:
            self.results["errors"] = "Invalid command"

    def get_results(self):
        '''
        Return calculated results
        '''
        return self.results

    def calculate_all_data(self):
        '''
        Find all process PID numbers for all tracked processes
        '''
        pid_list = []
        for pair in self.measured_proc:
            process_name, file_extension = pair
            pid_list = Process_tracker.all_processes_cmdline(process_name)
            #if there is no running process skip next step
            if pid_list:
                self.process_usage_pid(pid_list, file_extension)


    @staticmethod
    def all_processes_cmdline(process_name):
        '''
        Finds all processes with given proces name
        If there is none return [] 
        '''
        pids = list()
        for proc in psutil.process_iter():
            try:
                if proc.name() == process_name:
                    pids.append(proc.pid)
            except:
                #we tried to read a procces with forbidden acces
                continue
        return pids


    def process_usage_pid(self, pids, file_extension):
        '''
        Given list of pids, measure memory and cpu usage for each process
        '''
        for pid in pids:
            proc = psutil.Process(pid)
            #refactor command line for retrieveng script name from commandline (because python script name is always python.exe so thats why u need commandline)
            line = "\\".join(proc.cmdline())
            #ugly but works
            name = [one for one in re.split('[/\\\\]', line) if (file_extension in one and one not in self.ignored_proc)]
            if name:
                if name[0] in ['siwim_i.py', 'siwim_engine.exe']:
                    self.results[name[0].split(".")[0]] = {"CPU": proc.cpu_percent(interval=0.005) / psutil.cpu_count(), "MEM": round(proc.memory_full_info().uss / (1024 * 1024), 1)}
                else:
                    self.results['other_processes'].append({'process_name': name[0], "CPU": proc.cpu_percent(interval=0.005) / psutil.cpu_count(), "MEM": round(proc.memory_full_info().uss / (1024 * 1024), 1)})


    def get_drive_sizes(self):
        '''
        Gets all disk % usage and sizes
        '''
        drives = list()
        #array of disk names
        drive_names = list()
        for one in psutil.disk_partitions():
            #skip optical cdrom readers
            if one.opts != "cdrom":
                drive_names.append(one.mountpoint[0])
                disk = psutil.disk_usage(one.mountpoint)
                #total disk space in GB
                total_disk = round(disk.total / (1024 ** 3), 1)
                #disk space still free in GB
                free_disk = round(disk.free / (1024 ** 3), 1)
                #Percentage of utilized disk space
                util_disk = round(disk.used * 100 / disk.total, 2)
                drives.append({"name": one.mountpoint[0], "total": total_disk, "free": free_disk, "utilized": util_disk })
        self.results["drives"] = drives

        #check if any important disk is missing
        important_disks = json.loads(self.conf_file.get("disks", "important_disks"))
        for imp_disk in important_disks:
            if imp_disk not in drive_names:
                self.results["Disk_error"] = "{0} drive is missing".format(imp_disk)


    def get_sw_hw_data(self):
        '''
        Gets all software and hardware data from swmstat.xml file for easier following of all active version on all sites.
        '''
        tree = ET.parse(self.conf_file["conf"]["swmstat"])
        root = tree.getroot()

        #finding all the usefull data in xml
        self.results["python_version"] = root.find("python_version").text
        self.results["siwim_version"] = root.find("siwim_version").text
        self.results["updt_chan"] = root.find("update_channel").text
        self.results["e_version"] = root.find("siwim_e_version").text
        self.results["i_version"] = root.find("siwim_i_version").text

        cameras = set()
        for child in root.findall("./cameras/camera"):
            cameras.add(child.attrib["type"])
        self.results["cameras"] = ", ".join(sorted(cameras))


if __name__ == "__main__":
    type = sys.argv[1]
    tracker = Process_tracker(type)
    print(json.dumps(tracker.get_results()))