#!/usr/bin/python3
#version = 1.0.5.1

import socket, time, json, threading, configparser, os, math

class Log_line():

    def __init__(self, str_line):
        try:
            data = str_line.split("; ")
            self.date = data[0]
            self.timestamp = data[1]
            #type can be DEBUG INFO CRICITAL ERROR NOTICE
            self.type = data[2]
            #Which module sent log line
            self.sender = data[3]
            #log line message
            self.line = data[-1]
            #error state check if log line was full and succesfully splited in class elements
            self.error_state = False
        except:
            print("CANT PROCESS LINE:", str_line)
            self.line = ''
            self.date = ''
            self.error_state = True

    
    def get_type(self):
        '''
        Returns log line type
        '''
        return self.type

    def get_error_state(self):
        '''
        Returns log error state (False when log couldnt save in class)
        '''
        return self.error_state

    
    def get_msg_line(self):
        '''
        Returns message line of a log
        '''
        return self.line


    def get_sender(self):
        '''
        Returns log line sender
        '''
        if "tsplit" in self.sender or "weigh" in self.sender:
            return "".join(x for x in self.sender.split(":")[0] if not x.isdigit())
        return self.sender


class Log_zabbix():

    def __init__(self):
        self.conf_file = configparser.ConfigParser()
        self.conf_file.read("D:\zabbix_scripts\conf.ini")
        self.listen_port = int(self.conf_file['communication_ports']['log_sender'])
        self.lstn_sock = None
        self.status_port = int(self.conf_file['communication_ports']['info_sender'])
        self.ignored_types = json.loads(self.conf_file.get("log_data", "ignore_list"))
        self.reset_data_dict()

        #For saving critical line that occured
        self.critical_line = False
        self.error_line = False
        #counts how many lines happend after critical error and still no restart happend 
        self.lines_after_critical = 0
        self.log_type_dict = dict()

        zabbix_thread = threading.Thread(target=self.zabbix_update_request)
        zabbix_thread.start()

        self.start_log_listening()

    def connect_log_port(self):
        '''
        Connects to port that send siwim_i logs. If connection is unavailable retries every 5 secs
        '''
        connected = False
        print(f"Trying to connect to port {self.listen_port}")
        self.lstn_sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        while not connected:
            try:
                #try to connect to port
                self.lstn_sock.connect(('127.0.0.1', self.listen_port))
                connected = True
                print(f"Connected to port {self.listen_port}... Listening")
            except:
                time.sleep(5)

        
    def reset_data_dict(self):
        '''
        Resets data dictionary to default values.
        '''
        self.data = {'zeroing_data': dict(), 'invalid_data': dict()}
        self.log_type_dict = dict()


    def transfort_data_for_zabbix(self):
        '''
        Transforms all the data collected for zabbix format
        Because zabbix reads jsonpath different and data needs to bi in the right format
        '''

        #zeroing data from {'name': zeroed_number} ::> {'ch': name, 'z': zeroed_number}
        final_zero = list()

        already_used = list()
        for key, value in self.data['zeroing_data'].items():
            final_zero.append({'ch': key, 'z': value})
            already_used.append(key)
        
        for sender, chanels in self.data['invalid_data'].items():
            for name, times in chanels.items():
                name = f"Ch{name}"
                if name in already_used:
                    #we just add to the existing one
                    for one in final_zero:
                        if one['ch'] == name:
                            one.setdefault(sender[0], 0)
                            if sender[0] != "t":
                                one[sender[0]] += times
                            else:
                                one[sender[0]] += math.ceil(times / 4.0)
                else:
                    #make a new dict and add to te list
                    final_zero.append({'ch': name, sender[0]: times})
                    already_used.append(name)

        self.data['logs'] = list()
        for type, times in self.log_type_dict.items():
            if type not in self.ignored_types and "_" not in type and ":" not in type:
                self.data['logs'].append({"type": type, "occ": times})

        self.data['zeroing_data'] = final_zero
        del self.data['invalid_data']
        return json.dumps(self.data)


    def start_log_listening(self):
        '''
        Starts listnening to the log port and retrieving all the data.
        Than passing to other functions to check the level of importance of that log line
        '''
        lines_processed = 0
        while True:
            self.connect_log_port()

            while True:
                #Tries to receive log line, if not connection is probably closed
                try:
                    data = self.lstn_sock.recv(512).decode()
                except:
                    print(f"Connection to port {self.listen_port} was lost.")
                    break

                if lines_processed == 100:
                    print("100 log lines processed")
                    lines_processed = 0
                else:
                    lines_processed += 1

                log = Log_line(data.strip())
                #checks if log was successfully saved in class otherwise skip log analysis
                if log.get_error_state():
                    continue
                
                self.log_type_dict.setdefault(log.get_type(), 0)
                self.log_type_dict[log.get_type()] += 1

                if "SIGHUP" in log.get_msg_line():
                    self.log_type_dict.setdefault("SIGHUP", 0)
                    self.log_type_dict["SIGHUP"] += 1
                
                #count lines if critical happend
                if self.critical_line:
                    self.lines_after_critical += 1

                if "zeroed" in log.get_msg_line():
                    zeroed_channel = log.get_msg_line().split(" ")[0]
                    self.data['zeroing_data'].setdefault(zeroed_channel, 0)
                    self.data['zeroing_data'][zeroed_channel] += 1
                
                if "Invalid data" in log.get_msg_line():
                    invalid_channels = log.get_msg_line().split("channel(s) ")[-1].split(", ")[:-1]
                    for one in invalid_channels:
                        sender = log.get_sender()
                        self.data['invalid_data'].setdefault(sender, dict())
                        self.data['invalid_data'][sender].setdefault(one, 0)
                        self.data['invalid_data'][sender][one] += 1
                
                #if problems with E key appeares.
                if "check for presence of dongle" in log.get_msg_line():
                    self.critical_line = log.get_msg_line()

                #sigh that system will crash.
                if "bad allocation" in log.get_msg_line():
                    self.critical_line = log.get_msg_line()

                #if any module goes into emergency mode
                if "emergency mode" in log.get_msg_line():
                    self.critical_line = log.get_msg_line()

                #Check if log line is critical type
                if log.get_type() == "CRITICAL":
                    self.critical_line = log.get_msg_line()

                #Check if log line is error type
                if log.get_type() == "ERROR":
                    self.error_line = log.get_msg_line()
                
                #DISABLED because we want to catch all errors
                #Check if system was restarted and there is no more critical mode
                #if "9093 established" in log.get_msg_line():
                    #self.critical_line = False
                    #self.lines_after_critical = 0
                
    
    def zabbix_update_request(self):
        '''
        Waits for zabbix request and then sends current data.
        '''
        while True:
            updt_soc = socket.socket()
            host = '127.0.0.1'
            updt_soc.bind((host, self.status_port))
            updt_soc.listen(1)
            client, addres = updt_soc.accept()

            requested_info_type = client.recv(512).decode()
            #check which type of info is requested
            if requested_info_type == "summary":
                #all data gathered after last summary request
                message = self.transfort_data_for_zabbix()
                self.reset_data_dict()
            
            elif requested_info_type == "critical":
                #only data if critical log line appeared in logs
                if self.critical_line: # and self.lines_after_critical >= 1: currently disabled because we always want to see critical errors
                    message = self.critical_line
                    self.critical_line = False
                    self.lines_after_critical = 0
                else:
                    message = 'No critical error found'

            elif requested_info_type == "error":
                #only data if error log line appeared in logs
                if self.error_line:
                    message = self.error_line
                    self.error_line = False
                else:
                    message = 'No error line found'
                    
            elif requested_info_type == "versions":
                #Once a day for version tracking
                full_string = ""
                for file in filter(os.path.isfile, os.listdir(".")):
                    read = open(file, "r")
                    for _ in range(3):
                        version_line = read.readline()
                        if version_line[1:8] == "version":
                            only_version = version_line.strip().split(" = ")[-1]
                            full_string += "{0}: {1} \n".format(file, only_version)
                            read.close()
                            break
                message = full_string[:-1]
            else:
                message = "Request command not recoqnized"

            client.send(message.encode())
            updt_soc.close()
            print("Sent data to zabbix")


if __name__ == "__main__":
    log_reader = Log_zabbix()

