#!/usr/bin/python3
import ctypes
import datetime
import logging
import os
import platform
import re
import shutil
import signal
import subprocess
import sys
import threading
import time
import zipfile
from argparse import ArgumentParser

from consts import ROOT_PATH, SIWIM_E_PATH


def swmCurSite(loc):
    try:
        cfg = configparser.ConfigParser(strict=False)  # Ignore potential duplicate keys in conf
        cfg.read(loc)
        return cfg.get("global", "site")
    except:
        return "generic"


# Handler for the Interrupt signal with required parameters
def keyboard_interrupt(signum, frame):
    logger.info('Caught signal {} (shutdown)'.format(signum))
    global interrupted
    interrupted = True


def logLifeEvents(site_name, life_event):
    # if i had more time, i would write a shorter letter
    found_site = False
    # found_life_event = False
    if not os.path.exists("log/sys_diagnostics.xml"):
        if not os.path.exists("log/"):
            os.mkdir("log")
        diagnostics_tag = etree.Element("diagnostics")
    else:
        try:
            diagnostics_tag = etree.parse("log/sys_diagnostics.xml").getroot()
        except:
            pass
        diag_site_tags = diagnostics_tag.findall("site")
        for diag_site_tag in diag_site_tags:
            if diag_site_tag.attrib["name"] == site_name:
                try:
                    diag_site_tag.find(life_event).text = str(int(diag_site_tag.find(life_event).text) + 1)
                except:
                    diag_life_event_tag = etree.Element(life_event)
                    diag_life_event_tag.text = "1"
                    diag_site_tag.append(diag_life_event_tag)
                # found_life_event = True
                found_site = True
                break
    if not found_site:
        diag_site_tag = etree.Element("site")
        diag_site_tag.attrib["name"] = site_name
        diag_life_event_tag = etree.Element(life_event)
        diag_life_event_tag.text = "1"
        diag_site_tag.append(diag_life_event_tag)
        diagnostics_tag.append(diag_site_tag)
    fd = open("log/sys_diagnostics.xml", "w")
    fd.write(etree.tostring(diagnostics_tag, pretty_print=True).decode())
    fd.close()


def parse_module(module):
    typ = module.attrib.get('type')
    params = dict()
    module_children = module.getchildren()
    for module_child in module_children:
        value = None
        param_name = module_child.tag
        # first, let's get "the special cases" out of the way
        if typ == "rcv" and param_name == "offset_from_boss":
            value = dict()
            try:
                offsets_from_boss = module_child.findall("lane")
                for lane in offsets_from_boss:
                    mp = 1
                    try:
                        mp = int(lane.attrib["mp"])
                    except:
                        pass
                    if lane.attrib["num"] not in value:
                        value[lane.attrib["num"]] = dict()
                    value[lane.attrib["num"]][mp] = float(lane.text)
            except:
                continue
            if len(value) == 0:
                # consistency
                value = None
        #############################################
        elif typ == "pic" and (param_name == "default_cam_types" or param_name == "default_cam_names"):
            value = list()
            cam_infos = module_child.getchildren()
            for cam_info in cam_infos:
                value.append(cam_info.text)
            if len(value) == 0:
                # consistency
                value = None
        #############################################
        elif (typ == "out" or typ == 'ocli') and param_name == "filter_rules":
            # might aswell keep the default
            try:
                elements = module.find("filter_rules").findall("element")
                for element in elements:
                    if value == None:
                        value = list()
                    if "is_sibling_of" in element.attrib:
                        value.append((element.text, "is_sibling_of", element.attrib["is_sibling_of"]))
                    elif "not_sibling_of" in element.attrib:
                        value.append((element.text, "not_sibling_of", element.attrib["not_sibling_of"]))
                    else:
                        value.append((element.text,))
            except:
                continue
        #############################################
        elif typ == "cam" and param_name == "ignore_classes":
            class_lane_dict = dict()
            for lane in module_child.getchildren():
                classes = list()
                for cls in lane.getchildren():
                    try:
                        classes.append(int(cls.text))
                    except:
                        pass
                class_lane_dict[int(lane.attrib["num"])] = classes
            if len(class_lane_dict) != 0:
                value = class_lane_dict
        #############################################
        elif typ == "cam" and param_name == "fixed_offset":
            num_lane_dict = dict()
            for lane in module_child.getchildren():
                num_lane_dict[int(lane.attrib["num"])] = int(lane.text)
            if len(num_lane_dict) != 0:
                value = num_lane_dict
        #############################################
        elif typ == "cam" and param_name == "cam_focus":
            num_lane_dict = dict()
            for lane in module_child.getchildren():
                mp = 1
                try:
                    mp = int(lane.attrib["mp"])
                except:
                    pass
                if int(lane.attrib["num"]) not in num_lane_dict:
                    num_lane_dict[int(lane.attrib["num"])] = dict()
                num_lane_dict[int(lane.attrib["num"])][mp] = float(lane.text)
            if len(num_lane_dict) != 0:
                value = num_lane_dict
        #############################################
        elif typ == "cam" and param_name == "lanes":
            lanes = list()
            for lane in module_child.getchildren():
                lanes.append(int(lane.text))
            if len(lanes) != 0:
                value = lanes
        #############################################
        elif typ == "lrm" and param_name == "timeouts":
            timeouts = dict()
            timeout_nodes = list()
            try:
                timeout_nodes = module.find("timeouts").findall("timeout")
            except:
                pass
            impl_timeout_nodes = list()
            try:
                impl_timeout_nodes = module.find("timeouts").findall("impl_interval_timeout")
            except:
                pass
            if len(timeout_nodes) == len(impl_timeout_nodes):
                pairs = list(zip(timeout_nodes, impl_timeout_nodes))
            for pair in pairs:
                timeouts[pair[0].attrib["lane"]] = (pair[0].text, pair[1].text)
            if len(timeouts) != 0:
                value = timeouts
        #############################################
        elif typ == "can" and param_name == "board_ids":
            board_ids = list()
            try:
                board_id_nodes = module.find("board_ids").findall("board_id")
                for node in board_id_nodes:
                    board_ids.append(int(node.text))
            except:
                pass
            if len(board_ids) != 0:
                value = board_ids
        #############################################
        elif param_name == "downstream_modules":
            # handle this elswhere (for now)
            continue
        #############################################
        else:
            try:
                value = int(module_child.text)
            except:
                try:
                    value = float(module_child.text)
                except:
                    value = module_child.text
        if value != None:
            params[param_name] = value
    return params


if platform.system() == 'Windows' and platform.release() == '10':
    # Disable QuickEdit mode by setting it's flag to 0x00
    kernel32 = ctypes.WinDLL('kernel32', use_last_error=True)
    kernel32.SetConsoleMode(kernel32.GetStdHandle(-10), (0x4 | 0x80 | 0x20 | 0x2 | 0x10 | 0x1 | 0x00 | 0x100))
    del kernel32  # We don't need this variable floating around anymore, so we delete it

# Get the script parameters
parser = ArgumentParser()
parser.add_argument('-d', '--debug', nargs='?', const=logging.DEBUG, default=logging.INFO, type=int, help='run I with debug output')
parser.add_argument('-c', '--conf', default='conf/conf.xml', type=str, help='specify conf path')
parser.add_argument('-u', '--update', action='store_true', help='install libraries in requirements-{}.txt and update backup.py to newest version'.format(platform.python_version_tuple()[0]))
args = parser.parse_args()

if args.update:  # If script was run with update parameter, try to update libraries.
    if platform.system() == 'Linux':
        print('Automatic updating of libraries not (yet) implemented for Linux.')
    else:
        if sys.version_info[0] == 2:
            os.environ["PYTHONIOENCODING"] = "utf-8"
        try:
            subprocess.check_call('{} -m pip install --upgrade pip'.format(sys.executable))  # We have to set default Python encoding for this to work
        except (subprocess.CalledProcessError, LookupError) as e:
            print('Updating pip failed: {}.'.format(e))
        if platform.system() == 'Windows':
            try:  # Update siwim_backup
                PACKAGE_NAME = 'siwim_backup'
                SCRIPT_FOLDER = 'D:/siwim_mkiii/backup_script'
                SCRIPT = os.path.join(SCRIPT_FOLDER, 'backup.py')

                output = subprocess.check_output('{} -m pip download siwim-backup'.format(sys.executable)).decode()
                archive = re.findall(r'{}-\d\.\d.\d\.zip'.format(PACKAGE_NAME), output)[0]

                file_path = '{}/{}/backup_win.py'.format(archive.rsplit('.', 1)[0], PACKAGE_NAME)
                try:
                    os.makedirs(SCRIPT_FOLDER)
                except:  # We don't care if the file exists.
                    pass
                with zipfile.ZipFile(archive, 'r') as zip:
                    zip.extract(file_path)

                shutil.move(file_path, SCRIPT)
                os.remove(archive)
                shutil.rmtree(archive.rsplit('.', 1)[0])
            except Exception as e:
                print('Downloading siwim_backup failed: {}'.format(e))
        try:
            subprocess.check_call([sys.executable, '-m', 'pip', 'install', '-r', 'requirements-{}.txt'.format(sys.version_info[0])])
        except Exception as e:  # If this fails for whatever reason just keep on going ...
            print('Updating of libraries failed: {}'.format(e))
        os.execv(sys.executable, [sys.executable] + [f for f in sys.argv if f not in ['-u', '--update']])  # Restart with previous flags except for update flag.

try:  # Make sure all required libraries are installed. Appropriate requirements.txt is required. If a library which isn't included in any of these paths is the only one missing, just add an import to it anyways.
    import configparser
    from lxml import etree
    from communication_module import CommunicationModule
    import requests
    from py_logging.cestel_logging import init_logger, configure_default_logger, log, log_exception
    from packaging import version as pack_version
    from cestel_helpers.version import check_version_file
except Exception as e:
    print('Some libraries are missing. Attempting to update ... {}\n'.format(e))
    os.execv(sys.executable, [sys.executable] + (sys.argv + ['--update'] if '--update' not in sys.argv else sys.argv))  # Restart with --update parameter

if os.path.dirname(sys.argv[0]) != '':
    os.chdir(os.path.dirname(sys.argv[0]))
configure_default_logger(level=args.debug, console=args.debug)

# This is the logger that handles siwim_i file. It's one of two loggers (the other being update) that log both to file and console.
logger = init_logger('siwim_i', to_console=True)

interrupted = False

fd = None
modules = None
module_names = list()

end_i = False

update_host = ""
update_protocol = "https"
sites_path = ""

comm_module_name = ""

try:
    with open('.version', 'r') as f:
        version = re.findall(r'^\d+(?:\.\d+)+(?:\.|$)', f.readline().strip().replace('-', '.'))[0].rstrip('.').split('.')  # This regex returns an extra dot for partial versions.
    if not check_version_file('.version', logger_name='siwim_i'):
        logger.warning('Hashes don\'t match! Displayed version may be incorrect.')
except Exception as e:
    logger.error('Obtaining version failed: {}! Using legacy version which may be incorrect.'.format(e))
    version = [4, 13, 19]
channel = "stable"

logger.debug('Running in debug mode.')

try:
    fd = open(args.conf)
    doc = etree.parse(fd)
    fd.close()
    modules = list(doc.iter("module"))
    update_host = doc.find("update_host").text
    try:
        update_protocol = doc.find("update_protocol").text
    except:
        pass
    sites_path = doc.find("sites_path").text
    try:
        siwim_e_path = doc.find('siwim_e_path').text
    except:
        siwim_e_path = None
    try:
        channel = doc.find("update_channel").text
    except:
        pass
    logger.info("SiWIM-I v{} starting. Using {} update channel.".format(".".join(str(x) for x in version), channel))
    aggr_cnt = 0
    for module in modules:
        module_name = module.find("name").text
        if module_name in module_names:
            raise Exception("All module names must be unique!")
        module_names.append(module_name)
except:
    logger.exception('Parsing conf file failed:')
    sys.exit(0)

sites_path = sites_path.replace("\\", "/")
if sites_path[len(sites_path) - 1] == "/":
    sites_path = sites_path[:-1]
if sites_path == "" or not os.path.exists(sites_path):
    logger.critical("Sites directory doesn't exist. Exiting ...")
    sys.exit(0)

root_path = sites_path[0:sites_path.find("/sites")]

running = dict()
downstream = dict()

comm_started = False

site_name = swmCurSite((siwim_e_path if siwim_e_path else "{}/siwim_e".format(root_path)) + "/conf/siwim.conf")

# factory
for module in modules:
    params = parse_module(module)
    params["root_path"] = root_path
    name = params["name"]
    if module.attrib["type"] == "rcv":
        from receive_module import ReceiveModule

        params[SIWIM_E_PATH] = siwim_e_path  # Receive module needs to know where the conf is to obtain site information.
        params[ROOT_PATH] = root_path
        running[name] = ReceiveModule(params)
    elif module.attrib["type"] == "aggr":
        from aggregation_module import AggregationModule

        running[name] = AggregationModule(params)
    elif module.attrib["type"] == "cam":
        from camera_module import CameraModule

        running[name] = CameraModule(params)
    elif module.attrib["type"] == "pic":
        from picture_module import PictureModule

        running[name] = PictureModule(params)
    elif module.attrib["type"] == "out":
        from output_module import OutputModule

        running[name] = OutputModule(params)
    elif module.attrib['type'] == 'ocli':
        from output_client_module import OutputClientModule

        running[name] = OutputClientModule(params)
    elif module.attrib["type"] == "lrm":
        from alarm_module import AlarmModule

        running[name] = AlarmModule(params)
    elif module.attrib["type"] == "stat":
        params["update_channel"] = channel
        params["version"] = version
        from status_module import StatusModule

        running[name] = StatusModule(params)
    elif module.attrib["type"] == "ctu":
        from ctu_module import CtuModule

        running[name] = CtuModule(params)
    elif module.attrib["type"] == "can":
        from can_module import CanModule

        running[name] = CanModule(params)
    elif module.attrib["type"] == "hist":
        from hist_module import HistModule

        running[name] = HistModule(params)
    elif module.attrib["type"] == "comm":
        comm_module_name = name
        running[name] = CommunicationModule(params)
        comm_started = True
    elif module.attrib["type"] == "papago":
        from papago_module import PapagoModule

        running[name] = PapagoModule(params)
    elif module.attrib["type"] == "efoy":
        from efoy_module import EfoyModule

        running[name] = EfoyModule(params)
    # add downstreams the old way
    try:
        running[name].set_sites_path(sites_path)
        running[name].set_site_name(site_name)
        dm_list = list(module.iter("downstream_module"))
        downstream[name] = list()
        for dm in dm_list:
            downstream[name].append(dm.text)
    except:
        logger.exception('Got exception retrieving downstream_module:')

if not comm_started:
    name = "comm_default"
    comm_module_name = name
    downstream[name] = list()
    running[name] = CommunicationModule({"name": name, "root_path": root_path})
    running[name].set_sites_path(sites_path)
    running[name].set_site_name(site_name)

for key, module in running.items():
    logger.info("Starting " + key + " module ...")
    for downster in downstream[key]:
        try:
            module.add_downstream_module(running[downster])
        except:
            pass
            # register alarms HERE (moved from receive module's run)
    for ds_ky, ds_mod in module.downstream_modules_dict.items():
        try:
            ds_mod.register_alarm(module.name, module.typ)
        except:
            pass
    threading.Thread(target=module.run).start()

# Whenever Ctrl+C is pressed, sig_handler, which sets interrupted variable, will be called.
signal.signal(signal.SIGINT, keyboard_interrupt)
day = 0
while True:
    try:
        if end_i:
            logger.info('Shutting down.')
            for key1, module1 in running.items():
                module1.set_end()
            while True:
                all_dead = True
                for key1, module1 in running.items():
                    if module1.is_alive():
                        logger.debug('Waiting for {0} to die'.format(key1))
                        all_dead = False
                        break
                if all_dead:
                    logger.info('All threads have closed.')
                    break
                try:
                    time.sleep(0.5)
                except:
                    logger.info('Sleep failed?')
            logger.debug('Calling os._exit(0) on siwim_i.py')
            # logLifeEvents(site_name, "i_shutdowns")
            os._exit(0)
        for key, module in running.items():
            if not module.is_alive():
                try:
                    logger.info("Restarting " + key + " module ...")
                    threading.Thread(target=module.run).start()
                except:
                    logger.exception('Got exception while trying to restart a module:')
        loc = ""
        new_ver = pack_version.parse('.'.join([str(i) for i in version]))
        try:
            if day != datetime.datetime.now().day and update_host != "":
                logger.debug('Host: {0}, Channel: {1}'.format(update_host, channel))
                day = datetime.datetime.now().day
                resp = requests.post('{0}://{1}/check.php?sw=siwim-i&ch={2}'.format(update_protocol, update_host, channel), timeout=10).text
                resp_parts = resp.split()
                for part in resp_parts:
                    if part[0:4] == "ver=":
                        new_ver = pack_version.parse(part[4:])
                    if part[0:4] == "loc=":
                        loc = part[4:]
                        logger.debug('Update loc: {0}'.format(loc))
                loc = loc[loc.find("/"):]
                if new_ver.major > 4 and sys.version_info.major == 2:
                    logger.warning('SiWIM-I v5 does not support Python2. Upgrade to Python3 or use a different update channel. Update aborted.')
                elif pack_version.parse('.'.join([str(i) for i in version])) < new_ver:
                    end_i = True
        except requests.exceptions.ConnectTimeout:
            logger.warning('Update failed, because application could not connect to update host. If this problem persists, make sure a valid DNS is set and {} is reachable.'.format(update_host))
        except requests.exceptions.ConnectionError as e:  # Python2 doesn't have this error.
            logger.warning('Update failed: {}'.format(e))
        except:
            logger.exception('Updating exited with an unexpected error for url {0}://{1}/check.php?sw=siwim-i&ch={2}'.format(update_protocol, update_host, channel))
        if end_i:
            logger.info("Updating SiWIM-I to {}".format(new_ver))
            if not os.path.exists("update"):
                os.mkdir("update/")
            f = requests.get("{0}://{1}{2}".format(update_protocol, update_host, loc), stream=True)
            with open('update/update.zip', 'wb') as fd:
                for data in f.iter_content(chunk_size=1024):
                    fd.write(data)
            if new_ver.major == 4:
                # delete files to make sure update directory is ready for use
                up_files = os.listdir("update")
                for up_file in up_files:
                    if up_file != "update.zip" and up_file != "i_updater.py":
                        os.remove("update/" + up_file)
                zip_ref = zipfile.ZipFile("update/update.zip", 'r')
                zip_ref.extractall("update")
                zip_ref.close()
                try:
                    os.remove("update/update.zip")
                except WindowsError as e:
                    logger.error('Failed to delete update.zip: {}'.format(e))
                subprocess.Popen([sys.executable, "update/i_updater.py", str(os.getpid())])
            else:  # Updating works differently in SiWIM-I v5.
                zip_ref = zipfile.ZipFile('update/update.zip', 'r')
                zip_ref.extractall('.')
                zip_ref.close()
                shutil.rmtree('update')  # Delete update folder, since it's no longer relevant.
                subprocess.Popen([sys.executable, 'siwim_i.py'])  # This will run SiWIM-I v5 Launcher.
        elif running[comm_module_name].are_we_resetting_1():
            logger.info("Resetting I due to a frontend message ...")
            end_i = True
            subprocess.Popen([sys.executable, "update/i_updater.py", str(os.getpid())])
        elif interrupted:
            logger.info('Received signal interrupt.')
            end_i = True
        else:
            try:
                new_name = swmCurSite()
                # if name changed, reset i
                # generic indicates that something went wrong with site read
                # if name was generic from the get go, okay
                if site_name != new_name and new_name != "generic":
                    end_i = True
            except:
                pass
            try:
                time.sleep(1)
            except:
                logger.info('Exception during sleep (keyboard interrupt, maybe)')
    except NotImplementedError:  # If a NotImplementedError is raised and caught here, it's best to stop execution. This should only happen when siwim_i is restarting and previous instance doesn't quit properly.
        break
    except:
        logger.exception('Runner, general exception:')
