import datetime
import os
import re
import socket
import time

from generic_module import Module


class EfoyModule(Module):
    def __init__(self, args):
        expected_params = {"host": "192.168.4.10", "port": 10001, "polling_interval": 3600, "recv_buffer_size": 2048, "save_to_disk": 1}
        Module.__init__(self, args, expected_params)

    def generate_path(self, mod):
        pth = self.sites_path
        pth = pth + "/" + self.sname + "/ext"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + mod
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/efoy"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + self.name
        if not os.path.exists(pth):
            os.mkdir(pth)
        return pth

    def connect_tcp(self):
        self.s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        try:
            self.s.connect((self.host, self.port))
            self.s.settimeout(2)
            self.logger.info('Connected to {}:{}'.format(self.host, self.port))
            return True
        except socket.error as e:
            self.logger.warning('Connection to {0}:{1} refused ({2})'.format(self.host, self.port, e))
            return False
        except:
            self.logger.exception('Failed to connect:')
            return False

    def run(self):
        self.alive = True
        self.end = False
        try:
            sent = False
            received_at = time.time()
            while True:  # Main loop that's saving fuel level.
                if self.end:
                    self.alive = False
                    return
                line = str()
                fuel_level = str()
                # Only do this once per polling_interval.
                if not sent or time.time() - received_at > self.polling_interval:
                    if not self.connect_tcp():
                        self.logger.warning('Connection failed.')
                        self.alive = False
                        return

                    self.logger.debug('Sending cartridge request ...')
                    self.s.send(b'cartridge\r')
                    sent = True
                    try:
                        # Attempt to obtain data from edgar.
                        retries = 0
                        while retries < 15:  # Obtain the data.
                            received = self.s.recv(self.recv_buffer_size).decode()
                            self.logger.debug('Received response: {}'.format(received))
                            if received == str():
                                self.logger.warning('Received an empty string, exiting thread.')
                                self.alive = False
                                return
                            line = line + received.replace('\r', str())  # We have to remove \r for parsing in next step to work. TODO Refactor and optimize this.
                            self.logger.info('Response so far: {}'.format(line))
                            match = re.search(r'M28 \((\d+)%\)', line)
                            if match:
                                fuel_level = match.group(1)
                                received_at = time.time()
                                break
                            else:  # Full result hasn't been received yet.
                                retries += 1
                                continue
                        else:
                            self.logger.warning('Exiting thread {} due to unexpected response: {}'.format(self.name, line))
                            self.alive = False
                            return
                    except socket.timeout:
                        self.logger.warning('Connection timed out. Retrying ...')
                        time.sleep(60)
                        continue
                    except socket.error:
                        self.logger.debug('{0} with buffer [{1}] closed.'.format(self.s, self.recv_buffer_size))
                        self.logger.warning('Connection closed.')
                        self.alive = False
                        return
                    finally:
                        self.logger.debug('Closing socket.')
                        self.s.close()
                    for key, mod in self.downstream_modules_dict.items():
                        mod.set_upstream_info(self.get_name(), "fuel_level", fuel_level)
                    if self.save_to_disk != 0:
                        pth = self.generate_path("fuel")
                        tm = datetime.datetime.now().strftime('%Y-%m-%d-%H-%M-%S')
                        date = datetime.datetime.now().strftime('%Y-%m-%d')
                        header = "ts\tfuel_level[%]"
                        fd = open(pth + "/" + date + ".csv", "a+")
                        if not os.path.exists(pth + "/" + date + ".csv") or os.path.getsize(pth + "/" + date + ".csv") == 0:
                            fd.write(header + "\n")
                        fd.write(tm + "\t" + str(fuel_level) + "\n")
                        fd.close()
                time.sleep(1)
        except:
            self.logger.exception('General error:')
            self.alive = False
            return
