import datetime
import os
import socket
import struct
import time
from collections import deque

from generic_module import Module


class CanModule(Module):
    def __init__(self, args):
        expected_params = {"udp_rx_ip": "0.0.0.0", "udp_rx_port": 8172, "udp_tx_ip": "192.168.4.102", "udp_tx_port": 1234, "udp_e_ip": "127.0.0.1", "udp_e_port": 9000, "interval": 30, "root_path": None, "relevant_for_listeners": "t", "ap_fwver": 1,
                           "board_ids": [], "ap_uptime": 2, "polling_interval": 600, "mk_ip": "", "mk_port": 0}
        Module.__init__(self, args, expected_params)
        self.vdesc = ["unknown", "analog_12V", "analog_-12V", "analog_5V", "analog_-5V", "digital_12V"]
        self.tdesc = ["unknown", "T1__C", "T2__C", "T3__C", "T4__C", "Ti__C"]
        self.bname = ["unknown", "CAZ", "CAT", "SPS", "BRD4", "BRD5", "BRD6", "BRD7", "SAM1", "SAM2", "SAM3", "SAM4", "SAM5", "SAM6", "SAM7", "SAM8"]
        self.rx_s = None

    def generate_path(self, mod):
        pth = self.sites_path
        pth = pth + "/" + self.sname + "/ext"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + mod
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/can"
        if not os.path.exists(pth):
            os.mkdir(pth)
        pth = pth + "/" + self.name
        if not os.path.exists(pth):
            os.mkdir(pth)
        return pth

    def can_msg(self, idx, data):
        if len(data) > 8:
            raise ValueError("invalid data")
        d = bytearray(36)  # initializes elements to zero
        d[1] = 0x24  # = 36
        d[3] = 0x80
        d[21] = len(data)
        d[24:28] = struct.pack('>i', idx)  # > indicates big endian
        d[28:28 + len(data)] = data
        return d

    def brd_cmd(self, board_id, cmd, par1, par2):
        txsock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        txsock.settimeout(1)
        try:
            d = bytearray(8)
            d[0] = board_id
            d[1] = cmd
            d[2:4] = struct.pack('<i', par1)[:2]  # struct.pack returns 4 bytes, but we only care about first 2. < indicates little endian
            d[4:8] = struct.pack('<i', par2)
            txsock.sendto(self.can_msg(0xc0, d), (self.udp_tx_ip, self.udp_tx_port))
        finally:
            txsock.close()

    def dict2confstr(self, section, d):
        r = '[' + section + ']' + '\n'
        for k, v in d.items():
            r += k + '=' + v + '\n'
        return r

    def send_data(self, data):
        conf_str_data = self.dict2confstr("mydevice", data)
        sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        try:
            sock.sendto(conf_str_data.encode(), (self.udp_e_ip, self.udp_e_port))
        finally:
            sock.close()

    def bind_udp(self):
        self.rx_s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.rx_s.settimeout(1)
        try:
            self.rx_s.bind((self.udp_rx_ip, self.udp_rx_port))
            return 0
        except:
            self.logger.exception('Failed to bind socket')
            return 1

    def run(self):
        try:
            if self.bind_udp() == 0:
                command_queue = deque()
                start = time.time()
                polling_start = time.time()
                # implement this so I don't have to wait while testing
                # it won't harm the future either
                polling_interval = 0

                temp_dict = dict()
                volt_dict = dict()
                send_prompts = True
                while True:

                    if self.end:
                        try:
                            self.rx_s.close()
                        except:
                            self.logger.exception('Failed to close:')
                        self.alive = False
                        self.logger.debug('Thread closed correctly.')
                        return

                    # poll for data; create requests and send them serially
                    if time.time() - polling_start > polling_interval:
                        # once in a while, send at least one
                        send_prompts = True
                        for board_id in self.board_ids:
                            command_queue.append((board_id, self.ap_uptime, 0, 0))
                            command_queue.append((board_id, self.ap_fwver, 0, 0))
                            polling_interval = self.polling_interval
                        polling_start = time.time()
                    if len(command_queue) > 0 and send_prompts:
                        command = command_queue.popleft()
                        self.brd_cmd(command[0], command[1], command[2], command[3])
                        send_prompts = False
                    if len(command_queue) == 0:
                        send_prompts = True

                    # try to receive, data; except if failed
                    try:
                        data, addr = self.rx_s.recvfrom(64)
                        # whenever we receive something, send more prompts
                        send_prompts = True
                        try:
                            if self.mk_ip != "" and self.mk_port > 0:
                                self.fw_s.sendto(data, (self.mk_ip, self.mk_port))
                        except:
                            self.logger.exception("Didn't send to mk:")
                    except:
                        continue

                    # read data
                    data = bytearray(data)  # Does nothing in Python3 as return value is already bytearray.
                    dlc = data[21]
                    cid = struct.unpack('!I', data[24:28])[0]
                    cdt = data[28:36]
                    # voltage, do nothing
                    if cid == 0x90:
                        vid = data[28]
                        volt_dict[self.vdesc[int(vid)]] = format(struct.unpack('f', data[29:33])[0], '.2f')
                    # temperature
                    elif cid == 0x80:
                        tid = data[28]
                        temp_dict[self.tdesc[int(tid)]] = format(struct.unpack('f', data[29:33])[0], '.2f')
                        temp_dict["ts"] = datetime.datetime.now().strftime('%Y-%m-%d-%H-%M-%S')
                    # version
                    elif (cid >= 0xc1) and (cid <= 0xcf):
                        board_id = cdt[0]  # Since we have a bytearray, values are represented as numbers, so cast to int is irrelevant (and not Python3 compatible).
                        par_id = cdt[1]
                        if par_id == 1:  # fwver
                            fwver_major = cdt[3]
                            fwver_minor = cdt[2]
                            ver = '{0}.{1}'.format(fwver_major, fwver_minor)
                            for key, mod in self.downstream_modules_dict.items():
                                mod.set_upstream_info(self.get_name(), self.bname[board_id].lower() + "_firmware_version", ver)
                        elif par_id == 2:  # uptime
                            upt = struct.unpack('I', cdt[4:])[0]
                            self.logger.info(self.bname[board_id] + ' uptime: ' + str(upt))
                    else:
                        self.logger.error('unk ID ' + format(cid, '03x') + ' DATA ' + ' '.join(format(x, '02x') for x in cdt[:dlc]))

                    # forward voltage to downstream (specific analagoue values)
                    for key, mod in self.downstream_modules_dict.items():
                        for volt_key, val in volt_dict.items():
                            if volt_key != "unknown" and volt_key != "ts":
                                mod.set_upstream_info(self.get_name(), volt_key, str(val))

                    # save and forward temperature info once in an 'interval', if relevant
                    if time.time() - start > self.interval and "t" in self.relevant_for_listeners.lower():
                        pth = self.generate_path("temperature")
                        for key, mod in self.downstream_modules_dict.items():
                            mod.set_upstream_info(self.get_name(), "temperature_path", pth)
                        date = datetime.datetime.now().strftime('%Y-%m-%d')
                        fd = open(pth + "/" + date + ".csv", "a+")
                        header = "\t".join(self.tdesc[1:])
                        header = "ts" + "\t" + header
                        if not os.path.exists(pth + "/" + date + ".csv") or os.path.getsize(pth + "/" + date + ".csv") == 0:
                            fd.write(header + "\n")
                        tm = datetime.datetime.now().strftime('%Y-%m-%d-%H-%M-%S')
                        to_write = tm
                        for el in self.tdesc[1:]:
                            try:
                                to_write = to_write + "\t" + temp_dict[el]
                            except:
                                to_write = to_write + "\t" + "-273"
                        fd.write(to_write + "\n")
                        fd.close()
                        # -------------------
                        self.send_data(temp_dict)
                        start = time.time()
        except:
            # if ANYTHING happens, kill me softly
            self.logger.exception('General exception, warranting a module restart:')
            try:
                self.rx_s.close()
            except:
                self.logger.exception('Failed to close:')
            self.alive = False
            return
