#!/usr/bin/python3
#version = 2.2.1.0

from calendar import timegm
import xml.etree.ElementTree as xml_et
import time, configparser, os, sys, json, datetime, math, re
import tomlkit


class Vehicle_data:
    def __init__(self, time_interval):
        self.conf_file = configparser.ConfigParser()
        self.conf_file.read("D:\zabbix_scripts\conf.ini")
        self.active_site_name = self.get_site_name()
        self.num_of_lanes = self.get_number_of_lanes()
        self.current_time = datetime.datetime.now()
        self.results = {"site": self.active_site_name, "cams": list(), "lanes" : list(), "anpr" : list(), "other" : [dict()]}
        self.time_interval = time_interval * 60
        if self.check_i_version() == 4:
            self.camera_lanes, self.anpr_lanes, self.anprs, self.cam_storys, self.camera_info_V2, self.comark_info = self.camera_informations()
        else:
            self.camera_lanes, self.anpr_lanes, self.anprs, self.cam_storys, self.camera_info_V2, self.comark_info = self.camera_information_iv5()

        self.count_vehicles()
        self.count_pictures()
        self.count_pictures_V2()
        self.calculate_photos_per_vehicle()
        self.calculate_photos_per_vehicle_V2()
        self.count_comark()
        self.results["other"][0]["class_table"] = self.get_classification_table()

    def get_results(self):
        '''
        Returns all gathered results
        '''
        return self.results


    def get_site_name(self):
        '''
        Finds the latest site name on the system.
        '''
        conf_path = self.conf_file["conf"]["e_conf"]
        conf_parser = configparser.ConfigParser()
        conf_parser.read(conf_path)
        site_name = conf_parser.get('global', 'site')
        return site_name


    @staticmethod
    def most_recent_data(path, number_of_folders, extension):
        '''
        Return last "number_of_folders" files in given path
        '''
        os.chdir(path)
        files = sorted(filter(os.path.isfile, [f for f in os.listdir(".") if f.endswith(extension)]), key=os.path.getmtime)
        #gets the files with last "modified date" number depends of the parameter number of folders
        active_file = files[-number_of_folders:]
        return active_file
    

    def timestamp_in_interval(self, timestamp):
        '''
        Returns 0 if timestamp of a photo / vehicle is out of our observed interval
        Returns 1 if timestamp of a photo / vehicle is in our observed interval
        Returns -1 if timestamp of a photo / vehicle is in our observed interval by less than 10 seconds

        '''
        epoch_current_time = timegm(self.current_time.timetuple())
        #transforms timestamp to epoch
        epoch_timestamp = timegm(time.strptime(timestamp, "%Y-%m-%d-%H-%M-%S-%f"))

        time_distance = epoch_current_time - epoch_timestamp
        if time_distance <= 10:
            return -1
        return time_distance <= self.time_interval


    def get_number_of_lanes(self):
        '''
        Finds how many lanes are on specific site
        '''
        conf_path = self.conf_file["conf"]["site_conf"]
        conf_parser = configparser.ConfigParser()
        #reads conf
        conf_parser.read(conf_path.format(site_name=self.active_site_name))
        num_of_lanes = int(conf_parser.get('bridge', 'number_of_lanes'))
        return num_of_lanes
    

    def check_i_version(self):
        '''
        Check current i version running on system. Because we have a transition from iv4 to iv5 there is additional check of where swmstat.xml is located
        iv4 saves it in siwim_i location and iv5 saves it in active site folder
        '''
        i5_swmpath = self.conf_file["conf"]["iv5_swmstat"].format(site_name=self.active_site_name)
        if os.path.exists(i5_swmpath):
            return 5
        else:
            return 4

    def camera_informations(self):
        '''
        Returns camera names per all the lanes, to know what camera is filming what lane
        '''
        tree = xml_et.parse(self.conf_file["conf"]["i_conf"])
        root = tree.getroot()
        camera_info = dict() #{camera_1 : {"type": "xx", "story": int, ...}}

        results = dict()
        #active_cameras = list()
        #for a in root.findall("./modules/module/[@type='pic']"):
            #for cams in a.findall('default_cam_names/cam_name'):
                #active_cameras.append(cams.text)

        anpr_cams = list()
        storys = dict()
        for child in root.findall("./modules/module/[@type='cam']"):
            name = child.find('name').text
            camera_info[name] = dict()
            #if name in active_cameras:
            results[name] = list()
            camera_info[name]["lanes"] = list()
            for lanes in child.findall("./lanes/lane"):
                results[name].append(int(lanes.text))
                camera_info[name]["lanes"].append(int(lanes.text))
            
            if child.find("./picture_type").text == "anpr":
                anpr_cams.append(name)

            camera_info[name]["picture_type"] = child.find("./picture_type").text
            camera_info[name]["type"] = child.find("./init_type").text
            camera_info[name]["host"] = child.find("./host").text

            #finds camera story
            st = child.find("./story")
            if st != None:
                storys[name] = st.text
                camera_info[name]["story"] = st.text
            else:
                storys[name] = '0'
                camera_info[name]["story"] = '0'
        anpr_lanes = list()
        for one in anpr_cams:
            anpr_lanes += results[one]
        #anpr_lanes stays!

        comarks = dict()
        for child in root.findall("./modules/module/[@type='rcv']"):
            if child.find("./typ").text == "comark":
                name = child.find("./name").text
                com_lanes = set()
                for line in child.find("./offset_from_boss"):
                    com_lanes.add(line.attrib["num"])
                comarks[name] = {"lanes" : list(com_lanes)}
        #hardcoded anprs will definitely remove when iv5 will be launched
        anprs = {'ffgroup': anpr_lanes}
        return results, anpr_lanes, anprs, storys, camera_info, comarks
    

    def camera_information_iv5(self):
        '''
        Function that parses i conf file which are now toml and more organized.
        '''
        conf_path = self.conf_file["conf"]["iv5_conf"]
        with open(conf_path.format(site_name=self.active_site_name) + "/camera.toml", 'r') as file:
            toml_confs = tomlkit.parse(file.read())
        #get all cameras
        cameras = [section for section in toml_confs if "camera" in section]

        #----------------------------------------V1--------------------------------

        #finds watched lane for cameras
        camera_lanes = dict()
        #list for all anpr lanes
        anpr_lanes = list()
        #storys for each camera
        camera_story = dict()
        for cam in cameras:
            camera_lanes[cam] = toml_confs[cam]["lanes"]
            #check if camera is for anpr
            if "picture_type" in toml_confs[cam].keys() and toml_confs[cam]["picture_type"] == "anpr":
                anpr_lanes += toml_confs[cam]["lanes"]
            if "story" in toml_confs[cam].keys():
                camera_story[cam] = toml_confs[cam]["story"]
            else:
                camera_story[cam] = toml_confs["story"]

        #----------------------------------------------------------------------------
        # V2 in future only this will be used :D
        needed_data = ["lanes", "picture_type", "type", "host", "story"]
        camera_info = dict()
        for cam in cameras:
            one_dict = dict()
            #Fill dict with all needed data
            for data in needed_data:
                #if data is for camera specific
                try:
                    if data in toml_confs[cam].keys():
                        one_dict[data] = toml_confs[cam][data]
                    else: 
                        #data is in global
                        one_dict[data] = toml_confs[data]
                except:
                    raise Exception(f"{data} is mising from conf")
            camera_info[cam] = one_dict
        #----------------------------------------------------------------------------
        #Comark
        with open(conf_path.format(site_name=self.active_site_name) + "/acquisition.toml", 'r') as file:
            toml_confs = tomlkit.parse(file.read())

        comarks = dict()
        recieves = [section for section in toml_confs if "receive" in section]
        anprs = dict()
        #check in all receives if there is comark type
        for rec in recieves:
            if toml_confs[rec]["type"] == "comark":
                #we found comark now lets check what lanes are monitored
                comarks[rec] = {'lanes': list(toml_confs[rec]["offset_from_boss"].keys())}
            if toml_confs[rec]["type"] in ["ffgroup", "tecdetect4"]:
                anprs[toml_confs[rec]["type"]] = list(toml_confs[rec]["offset_from_boss"].keys())
        return camera_lanes, anpr_lanes, anprs, camera_story, camera_info, comarks


    def get_classification_table(self):
        '''
        Returns the name of classification table
        '''
        conf_path = self.conf_file["conf"]["classes_conf"]
        conf_parser = configparser.ConfigParser()
        #reads conf
        conf_parser.read(conf_path.format(site_name=self.active_site_name))
        return conf_parser.get('classes', 'classification_version')


    def count_pictures(self):
        '''
        Counts how many pictures every camera took in the time_interval
        '''
        camera_types = []
        #default camera picture path
        path_number = 1
        while True:
            try:
                cameras_path = self.conf_file["camera_data"]["path{0}".format(path_number)].format(site_name=self.active_site_name)
                os.chdir(cameras_path)
                camera_types += ["{0}\\{1}".format(cameras_path, name) for name in os.listdir('.') if os.path.isdir(name)]
                path_number += 1
            except:
                #There is no more camera paths in conf file
                break
        #calculate how many folder to check 
        chk_hour_folders = int(math.ceil(self.time_interval / 3600) + (1 if (self.time_interval / 60) % 60 > self.current_time.minute else 0))
        
        #check how many day folders we need to scan (+1 if the interval is larger that current day)
        chk_day_folders = int(math.ceil(chk_hour_folders / 24) + (1 if chk_hour_folders % 24 > self.current_time.hour else 0))

        list_of_camera_data = list()
        for typ in camera_types:
            os.chdir("{0}".format(typ))
            #Finds all cameras that are active (self.camera_lane(keys() present all cameras in i config)
            all_cameras = ["{0}\\{1}".format(typ, name) for name in os.listdir('.') if os.path.isdir(name) and "receive" not in name and name in self.camera_lanes.keys()]

            for one_camera in all_cameras:
                #counter for counting pictures
                counter = 0
                os.chdir("{0}".format(one_camera))
                dates = sorted(filter(os.path.isdir, os.listdir('.')))

                #check how many folder we need to check
                for one_day in dates[-chk_day_folders:]:

                    #check last hour in the folder
                    os.chdir("{0}\\{1}".format(one_camera, one_day))
                    hours = sorted(filter(os.path.isdir, os.listdir('.')))

                    #check how many folders we need to check depending on hour   
                    #room for improvment (if checking multiple days a bit to many folder is checked (not critical))
                    for one_hour in (hours if len(hours) < chk_hour_folders else hours[-chk_hour_folders:]):
                        os.chdir("{0}\\{1}\\{2}".format(one_camera, one_day, one_hour))
                        #sorts the pictures backwards (newest - oldest) for faster check (can skip when finds first older than our time frame)
                        pictures = sorted(filter(os.path.isfile, os.listdir(".")), reverse=True)

                        for one_pic in pictures:
                            timestamp = one_pic.split("_")[0]
                            #if picture is in our time interval
                            in_interval = self.timestamp_in_interval(timestamp)
                            if in_interval == -1:
                                continue
                            elif in_interval == 1:
                                counter += 1
                            else:
                                break
                #appends results for each camera
                list_of_camera_data.append({"name": "_".join(one_camera.split("\\")[-2:]), "photos": counter})

        for cam_data in list_of_camera_data:
            self.results["cams"].append(cam_data)


    def count_pictures_V2(self):
        '''
        Updated version of counting pictures. Now only cameras from i conf file will be checked. Names will be actualy their purpose and not manufacturer name
        '''
        camera_types_V2 = []
        #default camera picture path
        photo_path = self.conf_file["camera_data"]["photo_path"].format(site_name=self.active_site_name)

        #ips of cameras with anpr for changing camera type from overview to frontal
        ip_anpr = list()
        for camera in self.camera_info_V2.keys():
            type = self.camera_info_V2[camera]["picture_type"]
            man = self.camera_info_V2[camera]["type"]
            ip = self.camera_info_V2[camera]["host"]
            if type != "anpr":
                camera_types_V2.append("{0}\\{1}\\{2}".format(photo_path, man, camera))
            else:
                ip_anpr.append(ip)

        #calculate how many folder to check 
        chk_hour_folders = int(math.ceil(self.time_interval / 3600) + (1 if (self.time_interval / 60) % 60 > self.current_time.minute else 0))
        
        #check how many day folders we need to scan (+1 if the interval is larger that current day)
        chk_day_folders = int(math.ceil(chk_hour_folders / 24) + (1 if chk_hour_folders % 24 > self.current_time.hour else 0))
        list_of_camera_data = list()
        for one_camera in camera_types_V2:
            #counter for counting pictures
            counter = 0

            num_of_pictures = dict() #dictionary for counting pictures by lane
            camera_lanes = self.camera_info_V2[one_camera.split("\\")[-1]]["lanes"]
            for lane_num in camera_lanes:
                num_of_pictures[lane_num] = 0

            try: #if there is no folder there are no photos
                os.chdir("{0}".format(one_camera))
                dates = sorted(filter(os.path.isdir, os.listdir('.')))
            except:
                dates = []

            #check how many folder we need to check
            for one_day in dates[-chk_day_folders:]:

                #check last hour in the folder
                os.chdir("{0}\\{1}".format(one_camera, one_day))
                hours = sorted(filter(os.path.isdir, os.listdir('.')))

                #check how many folders we need to check depending on hour   
                #room for improvment (if checking multiple days a bit to many folder is checked (not critical))
                for one_hour in (hours if len(hours) < chk_hour_folders else hours[-chk_hour_folders:]):
                    os.chdir("{0}\\{1}\\{2}".format(one_camera, one_day, one_hour))
                    #sorts the pictures backwards (newest - oldest) for faster check (can skip when finds first older than our time frame)
                    pictures = sorted(filter(os.path.isfile, os.listdir(".")), reverse=True)

                    for one_pic in pictures:
                        data = one_pic.split("_")
                        timestamp = data[0]
                        lane = int(data[1])
                        #if picture is in our time interval
                        in_interval = self.timestamp_in_interval(timestamp)
                        if in_interval == -1:
                            continue
                        elif in_interval == 1:
                            counter += 1
                            num_of_pictures[lane] += 1
                        else:
                            break
            #appends results for each camera

            for lane in camera_lanes:
                cam_name = one_camera.split("\\")[-1]
                cam_type = self.camera_info_V2[cam_name]["picture_type"] if self.camera_info_V2[cam_name]["host"] not in ip_anpr else "frontal"
                if cam_type != "anpr":
                    list_of_camera_data.append({"name": "lane {0}: {1}".format(lane, cam_type), "photos": num_of_pictures[lane]})

        self.results["camsV2"] = list()
        for cam_data in list_of_camera_data:
            self.results["camsV2"].append(cam_data)
        
    
    @staticmethod
    def parse_single_tag(line, start, end):
        '''
        Parses a xml text line, finds and returns value in single tag.
        '''
        raw_data = line.split(end)[0].split(start)[-1]
        return raw_data


    def count_vehicles(self):
        '''
        Counts how many events happend in the last time_interval minutes
        '''
        #calculates how many xml-s to open. depending on what time is ti 
        chk_xml_files = int(math.ceil(self.time_interval / 3600 / 24) + (1 if (self.time_interval / 3600) % 24 > self.current_time.hour else 0))

        #check if save file exists othervise its 2 chains so its save1, save2
        if os.path.exists(self.conf_file["xml"]["e_xml_path"].format(site_name=self.active_site_name)):
            xml_dir = self.conf_file["xml"]["e_xml_path"].format(site_name=self.active_site_name)
            xml_paths = [xml_dir + '\\' + file for file in self.most_recent_data(xml_dir, chk_xml_files, '.xml')]
        else:
            xml_paths = list()
            for i in range(1, 3):    
                xml_dir = self.conf_file["xml"]["e_xml_path"].format(site_name=self.active_site_name)  + "{0}".format(i)
                xml_paths += [xml_dir + '\\' + file for file in self.most_recent_data(xml_dir, chk_xml_files, '.xml')] 

        lane_results = [0] * self.num_of_lanes
        all_unclassified = 0

        #weight monitoring
        under_five = [0] * self.num_of_lanes
        #we will check first axis of 113 vehicles
        first_axis_sum = [0] * self.num_of_lanes
        #how many vehicles 113
        weight_veh = [0] * self.num_of_lanes

        for one_xml in xml_paths:
            reader = open(one_xml)
            line = reader.readline()
            while line != "":
                rstr_line = line.lstrip()
                if ("<vehicle>" in rstr_line[0:9]):
                    timestamp = Vehicle_data.parse_single_tag(rstr_line, "<ts>", "</ts>")
                    in_interval = self.timestamp_in_interval(timestamp)
                    if in_interval == 1:
                        #in our interval and all vehicles after are ok
                        classification = Vehicle_data.parse_single_tag(rstr_line, "<cls>", "</cls>")
                        if classification == "140":
                            all_unclassified += 1
                            
                        lane = Vehicle_data.parse_single_tag(rstr_line, "<lane>", "</lane>")
                        lane_results[int(lane) - 1] += 1

                        #weight checks
                        weight = float(Vehicle_data.parse_single_tag(rstr_line, "<gvw>", "</gvw>")) / 9.81
                        axl_grps = Vehicle_data.parse_single_tag(rstr_line, "<axgrps>", "</axgrps")
                        if weight < 5:
                            under_five[int(lane) - 1] += 1

                        if axl_grps == "113" and 35 < weight:
                            #getting first axel weight
                            all_weights = Vehicle_data.parse_single_tag(rstr_line, "<acws>", "</acws")
                            first_axs = float(all_weights.split("</w>")[0].replace("<w>", "")) / 9.81
                            weight_veh[int(lane) - 1] += 1
                            first_axis_sum[int(lane) - 1] += first_axs

                    elif in_interval == -1:
                        break

                line = reader.readline()
            reader.close()

        all_vehicles = sum(lane_results)
        self.results["other"][0]["unclass"] = all_unclassified
        self.results["other"][0]["vehicles"] = all_vehicles

        #now check number of vehicles after siwim_i
        xml_dir = self.conf_file["xml"]["i_xml_path"].format(site_name=self.active_site_name)
        xml_paths = [xml_dir + '\\' + file for file in self.most_recent_data(xml_dir, chk_xml_files, '.xml')]

        #Used for counting all vehicles after i
        all_after_i = 0
        anpr_lanes = [0] * self.num_of_lanes
        anpr_dataV2 = {f"lane {lane}": {an : 0 for an in self.anprs.keys()} for lane in range(1, self.num_of_lanes + 1)}
        anpr_dataV3 = {f"lane {lane}": {an : 0 for an in self.anprs.keys()} for lane in range(1, self.num_of_lanes + 1)}
        for one_xml in xml_paths:
            #parsing xml
            reader = open(one_xml)
            line = reader.readline()
            while line != "":
                rstr_line = line.lstrip()
                if ("<vehicle>" in rstr_line[0:9]):
                    timestamp = Vehicle_data.parse_single_tag(rstr_line, "<ts>", "</ts>")
                    in_interval = self.timestamp_in_interval(timestamp)
                    lane = Vehicle_data.parse_single_tag(rstr_line, "<lane>", "</lane>")
                    if in_interval == 1:
                        #in our interval and all vehicles after are ok
                        all_after_i += 1
                        #checks if anpr was taken for that vehicle
                        if "<anpr>" in rstr_line:
                            anpr_lanes[int(lane) - 1] += 1

                        #Check all anpr recognitions (can be multiple (ffgroup, tecdetect))
                        #additional_data = rstr_line.split("<\wim>")[-1].split("<")
                        #for ext_data in additional_data:
                            #we check if the line is for anpr and contains a type=
                            #if ext_data.strip().startswith("anpr") and "type=" in ext_data:
                                #we find where did we get anpr 
                                #data_type = ext_data.split('type="')[1].split('"')[0]
                                #anpr_dataV2[f"lane {lane}"][data_type] += 1
                        

                        # Check all anpr recognitions (can be multiple (ffgourp, tecdetect))
                        pattern = re.compile(r'<anpr[^>]*type="(.*?)">(.*?)</anpr>', re.DOTALL)
                        matches = pattern.findall(rstr_line)
                        for match in matches:
                            anpr_type, nested_content = match
                            if nested_content.find("<lp>") != -1:
                                anpr_dataV2[f"lane {lane}"][anpr_type] += 1
                            
                    elif in_interval == -1:
                        break                               
                line = reader.readline()
            reader.close()
        self.results["other"][0]["veh_after_i"] = all_after_i
        self.results["other"][0]["all_anpr"] = sum(anpr_lanes)

        for one_lane in range(len(lane_results)):
            if str(one_lane + 1) in self.anpr_lanes:
                self.results["anpr"].append({"lane": "lane {0}".format(one_lane + 1), "anprs": anpr_lanes[one_lane]})
            self.results["lanes"].append({"name": "lane {0}".format(one_lane + 1), "vehicles": lane_results[one_lane],
                                          "first_axis": round(first_axis_sum[one_lane], 2), "113_axs": weight_veh[one_lane], "under_5t": under_five[one_lane]})
            
        #prepare dictionary for keeping track of detected anprs
        anprV2 = list()
        for anpr_lane in self.anpr_lanes:
            #for every lane we check if we have anpr camera there
            for anpr_type in self.anprs.keys():
                if str(anpr_lane) in self.anprs[anpr_type]:
                    #we have anpr camera there so we save how many vehicles were detected
                    data_dict = {"name": "lane {0}: {1}".format(anpr_lane, anpr_type), "veh": anpr_dataV2["lane {0}".format(anpr_lane)][anpr_type], "watch": "lane {0}".format(anpr_lane)}
                    anprV2.append(data_dict)

        self.results["anprV2"] = anprV2

    def calculate_photos_per_vehicle(self):
        '''
        Calculates how many photos did camera do per vehicle
        '''
        for each_camera in self.results['cams']:
            #change name from type_camera_# to camera_# to match them in i config file
            name = '_'.join(each_camera['name'].split("_")[1:])
            if name in self.camera_lanes.keys():
                watched_lanes = self.camera_lanes[name]
                all = sum([lane['vehicles'] for lane in self.results['lanes'] if lane['name'].split(" ")[1] in watched_lanes])
                each_camera['vehicles'] = all
                #calculate effectivnesess of a camera (value between 0 and 1)
                #if camera has 1 effectiveness all vehicles have exactly correct amount of pictures (story * 2 + 1)
                story = int(self.cam_storys[name])
                each_camera['effectiveness'] = round(each_camera['photos'] / all / (story * 2 + 1), 2) if all != 0 else 1
    

    def calculate_photos_per_vehicle_V2(self):
        '''
        Calculates how many photos did camera do per vehicle
        '''
        for each_camera in self.results['camsV2']:
            lane = each_camera["name"].split(":")[0].split(" ")[1]
            for res_lane in self.results['lanes']:
                if lane == res_lane['name'].split(" ")[1]:
                    lane_vehicles = res_lane["vehicles"]
                    break
            each_camera['vehicles'] = lane_vehicles
            #calculate effectivnesess of a camera (value between 0 and 1)
            #if camera has 1 effectiveness all vehicles have exactly correct amount of pictures (story * 2 + 1)
            s_lane, s_type = each_camera['name'].split(": ")
            for cam_name, cam_info in self.camera_info_V2.items():
                #searching for the right camera story
                if cam_info['picture_type'] == s_type:
                    if int(s_lane.split(" ")[-1]) in cam_info['lanes']:
                        story = int(cam_info['story'])
                        break
            each_camera['effectiveness'] = round(each_camera['photos'] / lane_vehicles / (story * 2 + 1), 2) if lane_vehicles != 0 else 1


    def count_comark(self):
        '''
        If commark exists on site counts all vehicles and also sorts them in classification ids
        '''
        
        #there is no comark on site so skip everything
        if self.comark_info == dict():
            return
        comark_path = self.conf_file["comark_data"]["path"].format(site_name = self.active_site_name)
        #if not os.path.exists(comark_path):
            #there is no comark on site so skip everything
            #return
        os.chdir(comark_path)
        chk_xml_files = int(math.ceil(self.time_interval / 3600 / 24) + (1 if (self.time_interval / 3600) % 24 > self.current_time.hour else 0))
        self.results["comark"] = list()

        #all_comarks = ["{0}\\{1}".format(comark_path, name) for name in os.listdir('.') if os.path.isdir(name)]
        #print(all_comarks)

        all_comarks = list()
        com_lanes = dict()
        for comarks in self.comark_info.keys():
            all_comarks.append("{0}\\{1}".format(comark_path, comarks))
            for l in self.comark_info[comarks]["lanes"]:
                com_lanes[l] = 0

        comark_lanes = [0] * self.num_of_lanes
        self.results["comarkV2"] = list()
        for one_comark in all_comarks:
            for data in self.most_recent_data(one_comark, chk_xml_files, '.xml'):
                #parsing xml
                reader = open(data)
                line = reader.readline()
                while line != "":
                    rstr_line = line.lstrip()
                    if ("<vehicle>" in rstr_line[0:9]):
                        timestamp = Vehicle_data.parse_single_tag(rstr_line, "time_iso_ms=", " speed").replace('"', "")
                        #transforming timestamp to classic format "%Y-%m-%d-%H-%M-%S-%f"
                        timestamp = timestamp.replace(":", "-").replace(".", "-").replace("T", "-")
                        in_interval = self.timestamp_in_interval(timestamp)
                        if in_interval == 1:
                            #in our interval and all vehicles after are ok
                            lane = Vehicle_data.parse_single_tag(rstr_line, "lane=", " lane_id").replace('"', "")
                            com_lanes[lane] += 1
                            comark_lanes[int(lane) - 1] += 1
                        
                        elif in_interval == -1:
                            break

                    line = reader.readline()
                reader.close() 
            self.results["comark"].append({"name": "comark_{0}".format(one_comark.split("\\")[-1]), "vehicles": sum(comark_lanes)})

        for lane, number in com_lanes.items():
            #find number of vehicles on that lane
            for res_lane in self.results['lanes']:
                if res_lane["name"] == "lane {0}".format(lane):
                    vehicles = res_lane["vehicles"]
                    break
            self.results["comarkV2"].append({"name": "lane {0}: comark".format(lane), "captured": number, "vehicles": vehicles})


            
if __name__ == "__main__":
    time_interval = int(sys.argv[1])
    data = Vehicle_data(time_interval)

    json_result = json.dumps(data.get_results())
    print(json_result)