import io
from collections import deque

import requests
from PIL import Image, UnidentifiedImageError
from requests.auth import HTTPDigestAuth

from abstract.module import Module
from consts import CAM_HOST, CAM_LANES, CAM_PASS, CAM_PIC_TYPE, CAM_PORT, CAM_USER, MOD_TYPE
from exceptions import NoPhoto, StopModule


class CameraCammra(Module):
    def __init__(self, args):
        self.vehicles_dict = {}
        # Define defaults that should be overwritten by the conf.
        self.lanes = []
        self.missed_count = 0
        self.uname = ''
        self.passw = ''
        self.max_wait_time = 0

        # Initialize the module, passing it a list of keys that must exist for it to work properly.
        Module.__init__(self, args, mandatory_keys=(CAM_HOST, CAM_LANES, CAM_PASS, CAM_PIC_TYPE, CAM_PORT, CAM_USER, MOD_TYPE))
        self.type = 'ffgroup'
        self.vehicles_dict = {}

    def add_vehicle(self, vehicle, recv_module_name):
        if recv_module_name not in self.vehicles_dict:
            self.vehicles_dict[recv_module_name] = deque()
        self.vehicles_dict[recv_module_name].append(vehicle)

    def get_photo(self, veh) -> None:
        try:
            veh_ts, _, lane = self.get_photo_data(veh, self.max_wait_time)
        except NoPhoto as e:
            self.logger.warning(e)
            return

        anpr_vehs = veh.findall('anpr')
        if len(anpr_vehs) == 0:
            raise NoPhoto('No anpr tag in vehicle.')
        for av in anpr_vehs:
            if av.find('cam_addr').text == self.host:
                anpr_veh = av
                break
        else:
            raise NoPhoto(f'Couldn\'t match cam_addr for {self.host}.')

        image_pth = f'http://{self.host}:{self.port}{anpr_veh.find("trg_ip").text}'
        photo_dir = ''
        if self.picture_type == 'anpr':
            photo_dir = 'lpr'
        elif self.picture_type == 'overview':
            photo_dir = 'photo'
        self.generate_data_dir(photo_dir, veh_ts)
        try:
            try:
                res = requests.get(image_pth, auth=HTTPDigestAuth(self.uname, self.passw))
            except requests.exceptions.ConnectionError as e:
                raise NoPhoto(f'Failed to get photo from {image_pth}: {e}.')
            img_name = f'{veh_ts}_{lane}_p00'
            if res.ok:  # If response is as expected.
                self.logger.debug("Saving image")
                if self.picture_type == "anpr":
                    coords = anpr_veh.find("coordinates").text[1:-1].replace(" ", "").split(",")
                    temp_buff = io.BytesIO()
                    temp_buff.write(res.content)
                    temp_buff.seek(0)
                    try:
                        im = Image.open(temp_buff)
                        im_c = im.crop((int(coords[0]), int(coords[1]), int(coords[0]) + int(coords[2]), int(coords[1]) + int(coords[3])))
                        im_c.save(f'{self.data_dirs[photo_dir]}/{img_name}.jpg', 'jpeg', optimize=True)
                    except UnidentifiedImageError:
                        with open(f'{self.data_dirs[photo_dir]}/{img_name}.dump', 'wb') as img:
                            img.write(res.content)
                        raise NoPhoto(f'Received image was not in correct format. Saving raw data as {img_name}.dump.')
                    except:
                        self.logger.exception("Save/crop exception: ")
                        raise NoPhoto(f'Failed to parse/crop image. Response contained: {res.content}')
                else:
                    with open(f'{self.data_dirs[photo_dir]}/{veh_ts}_{lane}_p00.jpg', 'wb') as f:
                        f.write(res.content)
                        self.logger.info('Image saved.')
            elif res.status_code == 404:  # If image is missing.
                self.logger.warning(f'No photo for {img_name}')
            else:
                self.logger.error(f'Camera returned status code: {res.status_code} and message: {res.text}')
        except:
            self.logger.exception('General exception:')

    def run(self) -> None:
        self.alive = True
        self.end = False

        try:
            self.camera_loop()
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
            self.alive = False
