#!/usr/bin/python3
# Martin Konečnik, http://git.siwim.si/cestel/siwim-i
# Script serves as a launcher to SiWIM-I v5. It takes care of updating, installing missing libraries and restarting core.py (formerly known as siwim_i.py).
import configparser
import os
import platform
import re
import shutil
import subprocess
import sys
from argparse import ArgumentParser
from logging import getLogger
from pathlib import Path
from zipfile import ZipFile

import config
from consts import CONF_DIR, CONF_GLOBAL, GLOBAL_ALLOW_ROLLBACK, GLOBAL_UPDATE_CHANNEL, LOGGER_MAIN, LOG_DIR


def update_libraries() -> bool:
    try:  # Check if any libraries are missing and update them.
        subprocess.run(f'{sys.executable} -m pip install -r {args.requirements}', check=True, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    except subprocess.CalledProcessError as e:
        getLogger().error(f'Updating libraries failed: {e.stderr}')
        getLogger().debug(f'Output of command: {e.stdout}')
        return False
    return True


if __name__ == '__main__':
    parser = ArgumentParser()
    parser.add_argument('-c', '--conf', default='conf/conf.xml', type=str, help='specify conf path')
    parser.add_argument('--convert_conf', action='store_true', help='try to convert configuration file from v4 (xml) to v5 (ini) format')
    parser.add_argument('--disable_core', action='store_true', help='application will not replace itself with core.py on finish')
    parser.add_argument('--download_dir', default='.', type=str, help='where new version should be downloaded; note that if this is overwritten, the old version of SiWIM-I will be launched after download')
    parser.add_argument('--init_site', action='store_true', help='pass if basic configuration should be created')
    parser.add_argument('--install_backup', action='store_true', help='application will download siwim_backup.py')
    parser.add_argument('-r', '--requirements', default='requirements.txt', type=str, help='name of the requirements file that should be used for checking library versions')
    parser.add_argument('--sites_dir', type=str, help='pass, if you want files to be written to a different folder')
    parser.add_argument('--siwim_e_conf', default=config.siwim_e_conf, type=str, help='siwim_e_conf that should be used to determine site name')
    parser.add_argument('--siwim_i_dir', type=str, help='pass if I should write configuration and log files to a different folder')
    parser.add_argument('-v', '--verbose', action='count', default=0, help='run with debug output')
    args, core_args = parser.parse_known_args()

    libraries_ok = False
    # There are multiple possible locations for pip conf, but since this needs to work on standardized systems I just picked one.
    pip_conf_path = Path(Path.home(), 'AppData/Roaming/pip/pip.ini') if platform.system() == 'Windows' else Path(Path.home(), '.pip/pip.conf')
    if pip_conf_path.exists():
        pip_conf = configparser.ConfigParser()
        pip_conf.read(pip_conf_path)
        try:
            # This could be made more robust, but reality is, that it has to work on systems, which are standardized anyway.
            if pip_conf.get('global', 'index-url', fallback=None) == f'https://{config.pip_server}/simple' or pip_conf.get('global', 'extra-index-url', fallback=None) == f'https://{config.pip_server}/simple':
                libraries_ok = update_libraries()
            else:
                print(f'Make sure that settings in {pip_conf_path} contain section global and index-url=https://{config.pip_server}/simple. Updating of libraries skipped.')
                input('Press ENTER to close.')
                sys.exit(1)
        except configparser.NoOptionError:
            print(f'Pip configuration file found at {pip_conf_path}, but is missing index-url=https://{config.pip_server}/simple. Updating of libraries skipped.')
            input('Press ENTER to close.')
            sys.exit(1)
    else:
        print(f'Pip configuration not found at {pip_conf_path}. Updating of libraries skipped.')
        input('Press ENTER to close.')
        sys.exit(1)

    # Check if an update of SiWIM-I is required.
    if libraries_ok:
        import requests
        from cestel_helpers.log import init_logger
        from cestel_helpers.version import get_version
        from cestel_helpers.update import download_update
        from cestel_helpers.exceptions import ConfError, UpdateError
        from cestel_helpers.i_conf_manager import SEC_DEFAULT, convert_conf, read_conf
        from helpers.helpers import exit_with_prompt, get_current_site
        from packaging.version import InvalidVersion, parse

        # Obtain site name and define conf and log directories
        config.site_name = get_current_site(args.siwim_e_conf)
        if args.siwim_i_dir:
            config.conf_dir = Path(args.siwim_i_dir, CONF_DIR)
            config.log_dir = Path(args.siwim_i_dir, LOG_DIR)

        logger = init_logger(LOGGER_MAIN, folder=config.log_dir, log_to='launcher', level=20 - args.verbose * 10, console_level=20 - args.verbose * 10, to_console=True, no_date=True, add_line_number=True if args.verbose else False)
        FULL_VERSION = get_version()
        # TODO This should be addressed once there are no versions of SiWIM-I older than 5.0.0. It's purpose is to parse old devel versions in accordance with PEP440.
        try:
            PEP440_VERSION = str(parse(FULL_VERSION))
        except InvalidVersion:
            ver, index, commit = FULL_VERSION.split('-')
            PEP440_VERSION = f'{ver}.dev{index}+{commit}'

        APP_NAME = f'SiWIM-I v{FULL_VERSION} Launcher'

        logger.info(f'{APP_NAME} starting.')
        if core_args:
            logger.info(f'Arguments {core_args} not parsed and will be forwarded to core.py!')

        # Attempt to convert the configuration file if explicitly specified or if it's missing.
        if args.convert_conf or not Path(config.conf_dir, CONF_GLOBAL).is_file():
            try:
                logger.info(f'Parameter convert_conf was passed or no SiWIM-I v5 configuration found. Attempting to convert SiWIM-I v4 configuration file.')
                convert_conf(config.site_name, path=args.conf, out_dir=config.conf_dir)
                # TODO Maybe delete the old configuration?
                logger.info(f'Configuration successfully converted to SiWIM-I v5 and saved to {config.conf_dir}. It is recommended to check convert.log and report conversion failures to application maintainer.')
                # args.convert_conf = False  # We don't want to restart ad infinitum.
            except:  # TODO Disable updates and revert SiWIM-I version?
                logger.exception('Converting configuration file failed! SiWIM-I v5 can not work with SiWIM-I v4 configuration file! Exiting. Contact maintainer and/or revert to SiWIM-I v4.')
                exit_with_prompt()

        try:
            conf_global = read_conf(Path(config.conf_dir, CONF_GLOBAL))[SEC_DEFAULT]
        except ConfError as e:
            logger.critical(f'Invalid conf at {config.conf_dir}: {e}')
            exit_with_prompt()
        channel = conf_global.get(GLOBAL_UPDATE_CHANNEL)
        rollback = conf_global.get(GLOBAL_ALLOW_ROLLBACK)

        try:
            logger.debug(f'Update channel: {channel}')
            if channel:  # Blank channel isn't considered a valid channel.
                if download_update('siwim-i', channel, 'siwim_i.zip', PEP440_VERSION, execute=False, allow_rollback=rollback):
                    logger.info('Latest version downloaded successfully. Unpacking ...')
                    with ZipFile('siwim_i.zip', 'r') as zf:
                        zf.extractall(args.download_dir)
                    logger.debug('Latest version unpacked successfully.')
                    logger.info('Updating libraries ...')
                    libraries_ok = update_libraries()
                    if not libraries_ok:
                        logger.critical('Failed to update libraries. Application may not function correctly.')
                    os.remove('siwim_i.zip')
                else:
                    logger.debug(f'No update available for channel {channel}.')
        except UpdateError as e:
            logger.warning(f'Updater exited with error: {e}')
        except (requests.exceptions.ConnectTimeout, requests.exceptions.ConnectionError):
            logger.warning('Update failed, because application could not connect to update host. If this problem persists, make sure a valid DNS is set and software server is reachable.')
        except:
            logger.exception(f'Updating exited with an unexpected error for channel {channel}.')

        if platform.system() == 'Windows' and args.install_backup:
            try:  # Update siwim_backup
                PACKAGE_NAME = 'siwim_backup'
                SCRIPT_FOLDER = 'D:/siwim_mkiii/backup_script'
                SCRIPT = os.path.join(SCRIPT_FOLDER, 'backup.py')

                output = subprocess.check_output(f'{sys.executable} -m pip download siwim-backup').decode()
                archive = re.findall(fr'{PACKAGE_NAME}-\d\.\d.\d\.zip', output)[0]

                file_path = f'{archive.rsplit(".", 1)[0]}/{PACKAGE_NAME}/backup_win.py'
                try:
                    os.makedirs(SCRIPT_FOLDER)
                except:  # We don't care if the directory exists.
                    pass
                with ZipFile(archive, 'r') as arch:
                    arch.extract(file_path)

                shutil.move(file_path, SCRIPT)
                os.remove(archive)
                shutil.rmtree(archive.rsplit('.', 1)[0])
            except Exception as e:
                logger.error(f'Downloading siwim_backup failed: {e}')
    else:
        getLogger().error('Libraries could not be updated! Application may not work correctly!')

    if not args.disable_core:
        # When run from core.py, the program name will be passed, and we don't want duplicates. We also don't want to convert ad infinitum.
        new_args = [arg for arg in sys.argv if arg != 'siwim_i.py' and arg != '--convert_conf']
        getLogger().debug(f'core.py started with arguments {new_args}')
        os.execv(sys.executable, [sys.executable, 'core.py'] + new_args)
