# Implementation of data acquisition from ffgroup.
import copy
import datetime
from typing import Dict

from lxml import etree

from consts import FFGROUP_VEH_END, MOD_TYPE, RCV_OFFSET, RCV_PORT, RCV_SAVE, SAVE_PATH
from exceptions import NoData, StopModule
from generic_module import Module


class AcquisitionFfgroup(Module):
    def __init__(self, args) -> None:
        # Initialize the module, passing it a list of keys that must exist for it to work properly.
        Module.__init__(self, args, mandatory_keys=(FFGROUP_VEH_END, MOD_TYPE, RCV_OFFSET, RCV_PORT, RCV_SAVE))
        self.type = 'ffgroup'

    def parse_data(self, data: Dict[str, str], cam_ip: str) -> None:
        """ Attempts to parse received blob and convert it to SiWIM friendly format.
        :param data: Received data.
        :param cam_ip: IP of camera.
        :return List of vehicles or None if an error occurred while parsing and unprocessed data.
        """
        # Without defining the start of the string a situation can occur where a nested dict is found as start point.
        vehicles = []
        data_dict = data
        vehicle_node = etree.Element('vehicle')
        vehicle_1_node = etree.Element('vehicle')
        anpr_node = etree.Element('anpr')
        anpr_node.set('source', self.name)
        anpr_node.set('type', 'ffgroup')
        anpr_1_node = etree.Element('anpr')
        vehicle_node.append(anpr_node)
        vehicle_1_node.append(anpr_1_node)
        id_node = etree.Element('id')
        ts_node = etree.Element('ts')
        lp_node = etree.Element('lp')
        trg_ip_node = etree.Element('trg_ip')
        cam_addr_node = etree.Element('cam_addr')
        country_node = etree.Element('country')
        confidence_node = etree.Element('confidence')
        vehicle_type_node = etree.Element('cls')
        axconfig_tiein_node = etree.Element('axconfig_tiein')
        lp_coordinates_node = etree.Element('coordinates')
        cam_addr_node.text = cam_ip
        lane_node = etree.Element('lane')
        # We try to map roiID (which starts from 1) to actual lanes, based on our offset
        if len(self.offset) > 0:
            # FIXME Always add the smallest lane number to total. This is a compromise and won't work if order of lanes is different between SiWIM and ffgroup.
            lane_node.text = str(int(data_dict['roiID']) - 1 + int(min(self.offset)))  # FIXME Make indices in offset int.
        id_node.text = data_dict['packetCounter']
        anpr_node.append(id_node)
        lp_node.text = data_dict['plateUTF8']
        ts = data_dict['capture_timestamp']
        ms = ts[-3:]
        ts = datetime.datetime.fromtimestamp(int(ts[:-3])).strftime('%Y-%m-%d-%H-%M-%S') + '-' + ms
        ts_node.text = ts
        trg_ip_node.text = data_dict['imagesURI'][1]
        location_node = etree.Element('location')
        location_node.text = 'front'
        if data_dict['plateCountry'] != 'XX':  # Don't add country information if it doesn't exist.
            country_node.text = data_dict['plateCountry']
            anpr_node.append(country_node)
        confidence_node.text = data_dict['plateConfidence']
        anpr_node.append(confidence_node)
        try:
            vehicle_type_node.text = data_dict['vehicle_info'].get('type')
        except KeyError:
            vehicle_type_node.text = 'N/A'
        anpr_node.append(vehicle_type_node)
        lp_coordinates_node.text = str(data_dict['plateCoordinatesRelative'])
        anpr_node.append(lp_coordinates_node)
        try:
            vehicle_info = data_dict['vehicle_info']['type']
            if vehicle_info == 'BUS':
                axconfig_tiein_node.text = 'B'
            else:
                axconfig_tiein_node.text = 'X'
            anpr_node.append(axconfig_tiein_node)
        except:
            pass
        if self.vehicle_end:
            add_to_offset_node = etree.Element('siwim_i_add_to_offset')
            add_to_offset_node.text = '0'
            location_node.text = 'rear'
            anpr_node.append(add_to_offset_node)
        anpr_node.append(location_node)
        anpr_node.append(ts_node)
        anpr_node.append(lp_node)
        anpr_node.append(lane_node)
        anpr_node.append(trg_ip_node)
        anpr_node.append(cam_addr_node)
        # anpr_1_node; save EVERYTHING
        for key, val in data_dict.items():
            node = etree.Element(key)
            node.text = str(val)
            anpr_1_node.append(node)
        vehicles.append(etree.tostring(vehicle_1_node, encoding='utf-8').decode())

        for key, mod in self.downstream_modules_dict.items():
            mod.add_vehicle(copy.deepcopy(vehicle_node), self.name)
        self.logger.info(f'Sent {ts}_{lane_node.text} downstream.')

    def run(self) -> None:
        try:
            self.alive = True  # TODO Remove this value once application uses threading's is_alive.
            self.end = False

            self.am_i_a_server = True
            if not self.bind_tcp():
                raise StopModule(msg=f'Binding to port {self.port} failed!')
            self.logger.info('Started server. Listening for connections ...')
            self.s.listen()
            conn = None
            buffer = b''
            while True:
                self.throttle()
                if self.end:  # Checks for end flag while connection is being established.
                    raise StopModule('Shutdown flag detected.', conn=conn)
                try:
                    conn, addr = self.accept_tcp()
                except TimeoutError as e:
                    self.logger.debug(e)
                    continue
                try:
                    while True:
                        received, buffer = self.acquire_data(conn, buffer, b'{"packetCounter"', b'\n}', 'json', block_buffer=20480, save_info={SAVE_PATH: 'anpr'})
                        if received == b'' and buffer == b'':  # Break the loop when no data is returned and buffer is exhausted.
                            break
                        self.parse_data(received, addr[0])
                except NoData as e:
                    self.logger.debug(e)  # Log the error.
                    buffer = b''  # Reset buffer.
                    continue
                except StopModule as e:
                    raise StopModule(msg=str(e))
                except:  # Log the datetime of the vehicle that had to be ditched.
                    self.logger.exception('Data lost:')
                    raise StopModule(conn=conn, msg='Data lost!')
        except StopModule as e:
            self.log_stop_module(e)
