import datetime
import io
import math
import re
import time
from abc import ABC, abstractmethod
from collections import deque
from pathlib import Path
from typing import Optional, Tuple

import piexif  # type: ignore
import requests
import tomlkit
from PIL import Image, ImageFile  # type: ignore
from lxml import etree  # type: ignore
from piexif import helper
from requests.auth import HTTPDigestAuth

from consts import TS_FORMAT_STRING
from exceptions import NoPhoto, StopModule
from generic_module import Module

ImageFile.LOAD_TRUNCATED_IMAGES = True  # This is required to save images that are truncated for whatever reason #FIXME Check if this is required by newer pip. Problem was detected in an axis module.


def find_first_match(pattern, source):
    try:
        match = next(re.finditer(pattern, source))
        return source[match.start():match.end()]
    except:
        return None


class CameraModule(Module, ABC):
    def __init__(self, args, mandatory_keys: Tuple[str, ...] = tuple(), optional_keys: Tuple[str, ...] = tuple()):
        # Define defaults that should be overwritten by the conf.
        self.offset = tomlkit.inline_table()
        self.fixed_offset = tomlkit.inline_table()
        self.host = None
        self.lanes = []
        self.max_time_diff = 0.2
        self.max_wait_time = 0
        self.passw = ''
        self.picture_type = None
        self.port = None
        self.protocol = "http://"
        self.story = 0
        self.type = None
        self.uname = ''

        # Initialize the module.
        Module.__init__(self, args, mandatory_keys=mandatory_keys, optional_keys=optional_keys)
        self.vehicles_dict = {}
        self.missed_count = 0
        self.location = "front"  # may change when first vehicle events arrive; to be implemented
        if self.story == 0 and 'story_before' in args:
            self.story = args['story_before']

    def add_vehicle(self, vehicle, recv_module_name):
        # sub_vehicle = vehicle.getchildren()[0]
        if recv_module_name not in self.vehicles_dict:
            self.vehicles_dict[recv_module_name] = deque()
        self.vehicles_dict[recv_module_name].append(vehicle)

    def datetime_to_timestamp(self, dtm):
        return datetime.datetime.strptime(dtm, TS_FORMAT_STRING).timestamp()

    def timestamp_to_datetime(self, ts):
        return datetime.datetime.fromtimestamp(ts).strftime(TS_FORMAT_STRING)[:-3]

    def download_and_save(self, cam_ev_list, dtm, ts, lane, auth) -> None:
        offsets = [math.fabs(ts - t[0]) for t in cam_ev_list]  # Calculate all offsets.
        dif = min(offsets)  # Get the lowest offset.
        mini = offsets.index(dif)  # Get the index of the offset.
        self.logger.debug(f'Calculated offsets: {offsets}.')
        if dif > self.max_time_diff:
            raise NoPhoto(f'No photos within {self.max_time_diff}s. Closest was {dif:.2f}s away')
        cnt = 0
        for i in range(mini - self.story, mini + self.story + 1):
            if 0 <= i < len(cam_ev_list):
                req_string = self.protocol + self.host + cam_ev_list[i][1]
                try:
                    if not auth:
                        res = requests.get(req_string, timeout=5)
                    else:
                        res = requests.get(req_string, auth=HTTPDigestAuth(self.uname, self.passw), timeout=5)
                    self.logger.debug(f'Request: {req_string}.', stacklevel=2)
                except (requests.exceptions.ReadTimeout, requests.exceptions.ConnectionError):
                    raise NoPhoto(f'Reached timeout for {req_string}.')
                if not res.ok:
                    self.logger.debug(f'Error message: {res.text}.', stacklevel=2)
                    raise NoPhoto(f'Camera responded with {res.status_code}.')

                result = res.content

                calc_id = i - mini
                str_id = str(abs(calc_id))
                if len(str_id) < 2:
                    str_id = "0" + str_id
                if calc_id < 0:
                    str_id = "n" + str_id
                elif calc_id >= 0:
                    str_id = "p" + str_id
                appendice = str_id

                img_ts = cam_ev_list[i][0]
                img_ts_str = str(img_ts)
                ms = img_ts_str[img_ts_str.find(".") + 1:len(img_ts_str)]
                while len(ms) < 3:
                    ms = ms + "0"
                our_ts = datetime.datetime.fromtimestamp(img_ts).strftime("%Y-%m-%d-%H-%M-%S") + "-" + ms
                # if mint != "":
                user_comment = helper.UserComment.dump("SiWIM_PHOTO_TS:" + our_ts)

                temp_buff = io.BytesIO()
                self.logger.debug(f'Photo size: {len(result)}.', stacklevel=2)
                temp_buff.write(result)
                temp_buff.seek(0)

                try:
                    im = Image.open(temp_buff)
                except:
                    self.logger_dump.error(f'Not valid photo: {result}.')
                    raise NoPhoto(f'Failed to parse photo. Saved to dump.')

                camera = find_first_match(b'CAM=\S+', result)
                exif_dict = {
                    "0th": {
                        piexif.ImageIFD.Artist: camera.replace(b'=', b':') if camera is not None else b''
                    },
                    "Exif": {
                        piexif.ExifIFD.UserComment: user_comment
                    }
                }
                exif_bytes = piexif.dump(exif_dict)

                photo_dir = 'photo'
                self.generate_data_dir(photo_dir, dtm)

                # Save the photo. If saving fails, save a binary blob instead.
                try:
                    im.save(Path(self.data_dirs[photo_dir], f'{dtm}_{lane}_{appendice}.jpg'), 'jpeg', exif=exif_bytes, quality="keep", optimize=True)
                except OSError as e:
                    with open(Path(self.data_dirs[photo_dir], f'{dtm}_{lane}.truncated'), 'wb') as blob:
                        blob.write(result)
                    raise NoPhoto(f'Failed to save {dtm}_{lane}_{appendice} as image: {e}. Original data saved as truncated.')
                except:
                    with open(Path(self.data_dirs[photo_dir], f'{dtm}_{lane}.blob'), 'wb') as blob:
                        blob.write(exif_bytes)
                    raise NoPhoto(f'Failed to save {dtm}_{lane}_{appendice} as image. Saving binary blob.')
            cnt += 1

    def wait_for_future(self, ts):
        now = time.time()
        cnt = 0
        while (now - ts - 3) < 0:
            self.logger.debug(f'Timestamp {ts} is in the future (current time: {now})')
            time.sleep(1)
            now = time.time()
            cnt += 1
            if cnt >= self.max_wait_time:
                if cnt > self.max_wait_time:
                    self.logger.warning(f'Count is higher than max_wait_time ({cnt} > {self.max_wait_time}). Report this warning to application maintainer.')
                break

    def apply_offsets(self, ts: float, lane: str, v: float, mp: int):  # Lane must be string or it'll cause "TypeError 'int' object is not iterable."
        if lane in self.fixed_offset:
            ts += self.fixed_offset[lane] / 1000
        try:
            if lane in self.offset:
                ts += self.offset[lane][mp] / v
        except IndexError:
            raise NoPhoto(f'mp {mp} on lane {lane} is undefined!.')
        return ts

    def get_photo_data(self, vehicles: etree.Element) -> Optional[Tuple[str, float, int]]:
        """ Approximates photo's timestamp based on settings. Also returns timestamp of the vehicle and lane.
        :param vehicles: Element object containing vehicles tag. Only first vehicle is checked.
        :returns vehicle timestamp, photo timestamp and lane.
        """
        # Extract the vehicle tag.
        sub_veh = vehicles.getchildren()[0]

        lane = int(sub_veh.find('lane').text)
        if lane not in self.lanes:
            raise NoPhoto(f'Lane {lane} is ignored.')

        # Determine measuring point. Defaults to 0. Format of admpsec is axle<lane><mp>.
        try:
            tmp = sub_veh.find('admpsec').text
            mp = int(tmp[-1]) - 1
        except:
            mp = 0
        # Obtain the rest of data.
        dtm = sub_veh.find('ts').text
        try:
            v = float(sub_veh.find('v').text)
        except AttributeError:
            raise NoPhoto(f'Speed is missing from {sub_veh.tag}.')
        # Convert event time to timestamp (seconds from epoch) and add offsets.
        self.logger.info(f'Processing {dtm}_{lane} on mp {mp} using speed {v}.', stacklevel=2)
        try:
            ts = self.apply_offsets(self.datetime_to_timestamp(dtm), str(lane), v, mp)
        except NoPhoto as e:
            raise NoPhoto(f'No photo for {dtm}_{lane}: {e}.')
        self.logger.debug(f'Photo ts: {self.timestamp_to_datetime(ts)}.', stacklevel=2)
        # Checking if ts is in the future and waiting accordingly
        self.wait_for_future(ts)
        return dtm, ts, lane

    @abstractmethod
    def get_photo(self, sub_veh):
        pass

    def main(self) -> None:
        """ Main method that handles general photo related tasks. """
        self.set_upstream_info(self.name, 'camera', {'name': self.name, 'type': self.type, 'ip': self.host, 'perspective': self.picture_type})
        while True:
            self.throttle()
            if self.end:
                self.alive = False
                raise StopModule('Thread closed correctly.')
            elif self.missed_count >= 10:
                self.missed_count = 0
                self.alive = False
                raise StopModule('Last 10 vehicles had no photo. Closing thread.')
            for key in self.vehicles_dict:
                if len(self.vehicles_dict[key]) > 0:
                    try:
                        sub_veh = self.vehicles_dict[key].popleft()
                    except IndexError:
                        self.logger.error(f'Couldn\'t pop from {self.vehicles_dict} for key {key}.')
                        time.sleep(1)
                        continue
                    try:
                        self.get_photo(sub_veh)
                        self.missed_count = -1  # If getting of photo succeeded, reset the countdown.
                    except NoPhoto as e:
                        self.logger.warning(f'Requesting photo failed: {e}', stacklevel=2)
                    except:
                        ts = sub_veh.find('wim/ts').text if sub_veh.find('wim/ts') is not None else None
                        if ts:
                            self.logger.exception(f'Unexpected error while obtaining photo for {ts}. Vehicle saved to dump', stacklevel=2)
                            self.logger_dump.error(f'{ts}: {etree.tostring(sub_veh).decode()}')
                        else:  # FIXME This should be solved by *not* having data without wim/ts sent to camera modules.
                            self.logger.debug(f'Camera received data without wim/ts. Ignoring.', stacklevel=2)
                            self.logger_dump.critical(f'Data without wim/ts: {etree.tostring(sub_veh).decode()}.', stacklevel=2)
                            self.missed_count -= 1
                    finally:  # Technically this should only be done when an exception occurs, but rather than handle it for every exception we do it here and set it to -1 rather than 0.
                        self.missed_count += 1
            time.sleep(0.25)
