# Implementation of requesting photos from mobotix cameras.
import json
import re

import requests

from acquisition.camera import CameraModule
from consts import CAM_HOST, CAM_LANES, CAM_OFFSET, CAM_PIC_TYPE, CAM_STORY, MOD_TYPE, RCV_OFFSET
from exceptions import NoPhoto, StopModule


class CameraMobotix(CameraModule):
    def __init__(self, args):
        # Initialize the module, passing it a list of keys that must exist for it to work properly.
        CameraModule.__init__(self, args=args, mandatory_keys=(CAM_HOST, CAM_LANES, CAM_PIC_TYPE, CAM_STORY, MOD_TYPE, RCV_OFFSET), optional_keys=(CAM_OFFSET,))

    def check_model(self, lowest_supported_version: int = 15):
        """ Check if the camera is found and is supported. """
        url = f'{self.protocol}{self.host}/control/'
        try:
            # Obtains page containing version information and parses it to extract model number of camera.
            response = requests.get(url).text
            try:
                cam_ver = int(re.findall(r'MOBOTIX M(\d+)', response)[0])
            except IndexError:
                self.logger.error(f'A response was received from {url}, but it contained no camera ID. Response dumped to logger_dump:ERROR.')
                self.logger_dump.error(f'{url}: {response}')
                raise StopModule(msg=f'No camera at {url}.')
            if cam_ver < lowest_supported_version:  # Returns the part in parentheses.
                self.restart_on_fail = False
                raise StopModule(msg=f'Cameras M{cam_ver} are no longer supported! Lowest supported version is M{lowest_supported_version}.')
        except requests.exceptions.ConnectionError:
            raise StopModule(msg=f'No response from {url}.')
        except Exception as e:
            self.logger.exception(f'Lookup for camera model at url {url} failed:')
            raise StopModule(msg=f'Unhandled exception: {e}.')

    # versions M15 and above
    def get_photo(self, veh) -> None:
        photos_per_sec = 5
        event_list_loc = "/control/event.jpg?output=alarmlist"
        event_img_loc = "/record/events/"

        # Extract relevant data from xml.
        try:
            veh_ts, photo_ts, lane = self.get_photo_data(veh)
        except NoPhoto as e:  # This occurs if lane is wrong.
            self.logger.warning(e)
            return

        num_events_to_parse = self.story * 2 + 1
        tsofs = num_events_to_parse / (2 * photos_per_sec)
        evurl = self.protocol + self.host + event_list_loc
        evurl += "&length=" + str(num_events_to_parse)
        evurl += "&searchbytime_start=" + str(photo_ts + tsofs)  # TODO; do I need to add bias?
        try:
            response = requests.get(evurl, timeout=5)
        except:
            raise NoPhoto('Failed to obtain list of events due to timeout.')
        if not response.ok:
            raise NoPhoto(f'Obtaining list of events failed with status {response.text}: {response.reason}')

        # json array of events
        html = response.content
        json_array = json.loads(html)
        cam_ev_list = []
        for jso in json_array:
            aiv = jso["alarmimage"]
            tsv = float(jso["timestamp"])
            if len(aiv) != 8:
                continue
            pth = event_img_loc + aiv[0:3] + "/" + aiv[3:6] + "/E00000.jpg"
            cam_ev_list.append((tsv, pth))
        self.logger.debug('Received that many events: ' + str(len(cam_ev_list)))
        if len(cam_ev_list) == 0:
            raise NoPhoto(f'{evurl} response: {html.decode()}')
        self.download_and_save(cam_ev_list, veh_ts, photo_ts, lane, False)

    def run(self):
        self.alive = True
        self.end = False
        try:
            self.check_model()
            self.main()
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
            self.alive = False
