# Implementation of requesting photos from mobotix cameras.
import json
import re
from collections import deque

import requests

from abstract.module import Module
from consts import CAM_HOST, CAM_LANES, CAM_MAX_DIFF, CAM_PIC_TYPE, CAM_SAMPLING_INTERVAL, CAM_STORY_AFTER, CAM_STORY_BEFORE, \
    META_CONF, META_MEASUREMENT, META_SPEED, META_TIMESTAMP, META_VEHICLE, MOD_TYPE, RCV_OFFSET
from exceptions import NoPhoto, StopModule
from helpers.helpers import get_relevant_tags_from_xml


class CameraMobotix(Module):
    def __init__(self, args):
        self.vehicles_dict = {}
        # Define defaults that should be overwritten by the conf.
        self.lanes = []
        self.missed_count = 0
        self.max_time_diff = 0.2
        self.uname = ''
        self.passw = ''
        self.protocol = 'http://'
        self.story = 0
        self.max_wait_time = 0
        self.model = 'N/A'
        # Photo requesting related
        self.story_before = 0
        self.story_after = 0
        self.sampling_interval = 1

        # Initialize the module, passing it a list of keys that must exist for it to work properly.
        Module.__init__(self, args, mandatory_keys=(CAM_HOST, CAM_LANES, CAM_PIC_TYPE, CAM_SAMPLING_INTERVAL, CAM_STORY_AFTER, CAM_STORY_BEFORE, MOD_TYPE, RCV_OFFSET), optional_keys=(CAM_MAX_DIFF,))
        self.type = 'mobotix'
        self.urls = {'list': f'{self.protocol}{self.host}:80/control/event.jpg?output=alarmlist&length=', 'get': f'{self.protocol}{self.host}:80/record/events/'}

    def add_vehicle(self, vehicle, recv_module_name):
        if recv_module_name not in self.vehicles_dict:
            self.vehicles_dict[recv_module_name] = deque()
        self.vehicles_dict[recv_module_name].append(vehicle)

    def add_vehicle(self, vehicle, recv_module_name):
        if recv_module_name not in self.vehicles_dict:
            self.vehicles_dict[recv_module_name] = deque()
        self.vehicles_dict[recv_module_name].append(vehicle)

    def check_model(self, lowest_supported_version: int = 15):
        """ Check if the camera is found and is supported. """
        url = f'{self.protocol}{self.host}/control/'
        try:
            # Obtains page containing version information and parses it to extract model number of camera.
            response = requests.get(url).text
            try:
                self.model = int(re.findall(r'MOBOTIX M(\d+)', response)[0])
            except IndexError:
                self.logger.error(f'A response was received from {url}, but it contained no camera ID. Response dumped to logger_dump:ERROR.')
                self.logger_dump.error(f'{url}: {response}')
                raise StopModule(msg=f'No camera at {url}.')
            if self.model < lowest_supported_version:  # Returns the part in parentheses.
                self.restart_on_fail = False
                raise StopModule(msg=f'Cameras M{self.model} are no longer supported! Lowest supported version is M{lowest_supported_version}.')
        except StopModule:
            raise
        except requests.exceptions.ConnectionError:
            raise StopModule(msg=f'No response from {url}.')
        except Exception as e:
            self.logger.exception(f'Lookup for camera model at url {url} failed:')
            raise StopModule(msg=f'Unhandled exception: {e}')

    # versions M15 and above
    def get_photo(self, veh) -> None:
        photos_per_sec = 5

        # Extract relevant data from xml.
        try:
            veh_ts, photo_ts, lane = self.get_photo_data(veh, self.max_wait_time)
        except NoPhoto as e:  # This occurs if lane is wrong.
            self.logger.warning(e)
            return

        num_events_to_parse = (max(self.story_before, self.story_after) + 1) * 2 * self.sampling_interval + 1
        tsofs = num_events_to_parse / (2 * photos_per_sec)
        try:
            response = requests.get(self.urls['list'] + f'{num_events_to_parse}&searchbytime_start={photo_ts + tsofs}', timeout=5)
        except:
            raise NoPhoto('Failed to obtain list of events due to timeout.')
        if not response.ok:
            raise NoPhoto(f'Obtaining list of events failed with status {response.text}: {response.reason}')

        # json array of events
        html = response.content
        json_array = json.loads(html)
        cam_ev_list = {}
        for jso in json_array:
            aiv = jso['alarmimage']
            tsv = float(jso['timestamp'])
            if len(aiv) != 8:
                continue
            cam_ev_list[tsv] = f'{self.urls["get"]}{aiv[0:3]}/{aiv[3:6]}/E00000.jpg'
        self.logger.debug(f'Received {len(cam_ev_list)} events.')
        if len(cam_ev_list) == 0:
            raise NoPhoto(f'{self.urls["list"]} response: {html.decode()}')
        metadata = get_relevant_tags_from_xml(veh.find('wim'), (META_CONF, META_MEASUREMENT, META_SPEED, META_TIMESTAMP, META_VEHICLE))
        self.download_and_save(cam_ev_list, veh_ts, photo_ts, lane, False, metadata=metadata)

    def run(self):
        self.alive = True
        self.end = False
        try:
            self.check_model()
            self.camera_loop()
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
            self.alive = False
