import socket
import time
from collections import deque
from datetime import datetime

from lxml import etree

import config
from abstract.module import Module
from consts import OCLI_HOST, OCLI_IDLE, OCLI_PORT, OCLI_SAVE, SAVE_FOOTER, SAVE_HEADER, SAVE_PATH, SAVE_SUFFIX
from exceptions import StopModule


class OutputClientModule(Module):
    def __init__(self, args):
        self.max_idle_interval = 10
        # Initialize the module, passing it a list of keys that must exist for it to work properly and a list of optional keys that are recognized by it.
        Module.__init__(self, args, mandatory_keys=(OCLI_HOST, OCLI_PORT, OCLI_SAVE), optional_keys=(OCLI_IDLE,))
        self.type = 'siwim_client'
        self.parser = etree.XMLParser(remove_blank_text=True)
        # client-thread indice, vehicle deque pairs
        self.vehicles = deque()
        self.s = None
        self.transmitting = False

    def set_end(self):
        self.end = True

    def add_vehicle(self, vehicle, module_name="irrelevant"):
        if len(vehicle.getchildren()) == 0:
            return
        if self.save_data == 1:
            save_info = {
                SAVE_PATH: 'output',
                SAVE_HEADER: f'<?xml version="1.0" ?>\n<swd version="1"><site><name>{config.site_name}</name><vehicles>',
                SAVE_FOOTER: '</vehicles></site></swd>',
                SAVE_SUFFIX: 'xml'
            }
            self.write_data(save_info, etree.tostring(vehicle, encoding='utf-8').decode())
        if self.transmitting:
            self.vehicles.append(vehicle)

    def run(self):
        self.alive = True
        self.end = False

        try:
            while True:
                self.throttle()
                if not self.connect_tcp(timeout=5):
                    if self.end:
                        raise StopModule(f'Connection to {self.host}:{self.port} failed.', conn=self.s)
                    self.vehicles.clear()
                    continue

                self.transmitting = True
                self.logger.debug('Transmitting data.')
                start = datetime.now()
                while True:
                    if (datetime.now() - start).total_seconds() > self.max_idle_interval:
                        self.logger.debug('Stopping, because max_idle_interval was reached.')
                        self.transmitting = False
                        break
                    if self.end:
                        self.alive = False
                        self.transmitting = False
                        raise StopModule('Shutdown flag detected.', conn=self.s)
                    if len(self.vehicles) > 0:
                        try:
                            vehicle = self.vehicles.popleft()
                            complete_xml = '<swd version="1"><site><name>' + config.site_name + '</name><vehicles>' + etree.tostring(vehicle, encoding='utf-8').decode() + '</vehicles></site></swd>'
                            to_send = etree.fromstring(complete_xml, self.parser)
                        except:
                            self.logger.exception('Failed form a sendable string from xml:')
                            time.sleep(0.5)
                            continue
                        try:
                            self.s.send(etree.tostring(to_send, encoding='utf-8', pretty_print=False))
                            self.logger.debug('Data sent successfully!')
                            start = datetime.now()
                        except socket.timeout:
                            self.logger.warning('Connection timed out, restarting connection:')
                            # restart connection
                            self.transmitting = False
                            break
                        except socket.error:
                            raise StopModule('Socket error, restarting module:')
                    time.sleep(0.5)
                time.sleep(2)
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
