# This file contains various helper functions that are used across the application.
import re
import sys
from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional, Tuple

import piexif
from cestel_helpers.aliases import Element
from cestel_helpers.exceptions import SiwimError
from cestel_helpers.siwim import Siwim
from piexif import helper

import config
from consts import CONF_BACKUP_DIR, CONF_DIR, LOG_DIR, META_CAMERA_MAKE, META_CAMERA_MODEL, META_CONF, META_MEASUREMENT, META_SPEED, META_TIMESTAMP, META_VEHICLE, TS_FORMAT_STRING


def get_current_site(siwim_e_conf: Path, set_i_dir: bool = True) -> str:
    """ Returns current site or 'generic' if site name can't be read.
    :param siwim_e_conf: Path to siwim.conf file.
    :param set_i_dir: Indicates whether directories for I should be created within the site folder.
    """
    siwim = Siwim(siwim_root=config.sites_dir.parent)
    siwim.conf_file_siwim = siwim_e_conf
    new_site = siwim.site
    if new_site is None:
        raise SiwimError(f'Site name not found in {siwim_e_conf}.')
    if new_site != config.site_name:  # If site changed, we have to update conf and logging directories.
        config.conf_backup_dir = Path(config.sites_dir, new_site, CONF_BACKUP_DIR)
        config.conf_backup_dir.mkdir(parents=True, exist_ok=True)
        if set_i_dir:
            config.conf_dir = Path(config.sites_dir, new_site, CONF_DIR)
            config.conf_dir.mkdir(parents=True, exist_ok=True)
            config.log_dir = Path(config.sites_dir, new_site, LOG_DIR)
            config.log_dir.mkdir(parents=True, exist_ok=True)
    return new_site


def get_customers(siwim_e_conf: Path) -> List[str]:
    siwim = Siwim(siwim_root=config.sites_dir.parent)
    siwim.conf_file_siwim = siwim_e_conf
    customers = siwim.conf_site.get('global', {}).get('customers', None)
    if customers:
        return customers.split(',')
    else:
        return []


def exit_with_prompt() -> None:
    """ Convenience function for asking user for input and then exit. """
    input('Press ENTER to close.')
    sys.exit(1)


def dict2confstr(section: str, d: Dict[str, str]) -> str:
    ini_string = f'[{section}]\n'
    for k, v in d.items():
        ini_string += f'{k}={v}\n'
    return ini_string


def find_first_match(pattern: bytes, source: bytes) -> Optional[bytes]:
    """ Finds the first occurrence of a pattern in source string and returns it.
    :param pattern: Pattern to search for.
    :param source: Source string to search for.
    :return: First occurrence of the pattern in source string or None if not found.
    """
    try:
        match = next(re.finditer(pattern, source))
    except StopIteration:
        return None
    return source[match.start():match.end()]


def datetime_to_timestamp(dtm) -> float:
    return datetime.strptime(dtm, TS_FORMAT_STRING).timestamp()


def timestamp_to_datetime(ts) -> str:
    return datetime.fromtimestamp(ts).strftime(TS_FORMAT_STRING)[:-3]


def filter_timestamps(items: List[float], center: int, interval: int, before: int, after: int) -> List[float]:
    """ Filters a list.
    :param items: List to filter.
    :param center: Index of the center.
    :param interval: Indicates how many items should be skipped.
    :param before: Max number of items before the center.
    :param after: Max number of items after the center.
    :return: Filtered items.
    """
    start_index = center - interval * before
    end_index = center + 1 + interval * after
    if start_index < 0:  # We need to start at the correct first index, if out of bounds.
        start_index = center % interval
    return items[start_index:end_index:interval]


def update_photo_metadata(data: bytes, ts: str, metadata: Dict[str, Any], use_pyexiv2: bool = False) -> bytes:
    """ Adds metadata to the photo.
    :param data: JPG we're updating.
    :param ts: Timestamp of the photo.
    :param metadata: Additional metadata that will be added to the photo.
    :param use_pyexiv2: If true, use pyexiv2 library to update metadata.
    :return: Updated JPG.
    """
    if use_pyexiv2:
        with pyexiv2.ImageData(data) as img:
            camera = find_first_match(b'CAM=\S+', data)
            img.modify_exif({
                'Exif.Image.Artist': camera.decode().replace('=', ':') if camera is not None else '',  # Deprecated
                'Exif.Photo.UserComment': f'SiWIM_PHOTO_TS:{ts}',  # Deprecated
                'Exif.Image.DateTime': ts,  # Replaces UserComment.
                'Exif.Image.Make': metadata.get(META_CAMERA_MAKE),
                'Exif.Image.Model': metadata.get(META_CAMERA_MODEL)
            })
            img.modify_xmp({
                'Xmp.xmp.Site.App': f'SiWIM-I',  # TODO Add version
                'Xmp.xmp.Site.Name': config.site_name,
                # 'Xmp.xmp.Site.Id': 'c400ad5099e8',  # TODO
                'Xmp.xmp.Site.ConfId': metadata.get(META_CONF),
                'Xmp.xmp.Site.MeasurementId': metadata.get(META_MEASUREMENT),
                # 'Xmp.xmp.Vehicle.Adr': '12345',  # TODO
                'Xmp.xmp.Vehicle.Id': metadata.get(META_VEHICLE),
                # 'Xmp.xmp.Vehicle.Licence': 'ABCDE12',  # TODO
                # 'Xmp.xmp.Vehicle.Offset': 0.0,  # TODO Evaluate if this is worth it and what it'd even be.
                'Xmp.xmp.Vehicle.Speed': metadata.get(META_SPEED),
                'Xmp.xmp.Vehicle.Timestamp': metadata.get(META_TIMESTAMP)  # Timestamp of the vehicle before processing.
            })
            return img.get_bytes()
    else:
        camera = find_first_match(b'CAM=\S+', data)
        with config.lock:
            exif_dict = {
                '0th': {
                    piexif.ImageIFD.Artist: camera.replace(b'=', b':') if camera is not None else b'',
                    piexif.ImageIFD.DateTime: ts,  # Replaces UserComment
                    piexif.ImageIFD.Make: metadata.get(META_CAMERA_MAKE, 'N/A'),
                    piexif.ImageIFD.Model: metadata.get(META_CAMERA_MODEL, 'N/A'),
                    piexif.ImageIFD.Software: 'SiWIM-I',
                    piexif.ImageIFD.ImageHistory: config.site_name,
                },
                'Exif': {
                    piexif.ExifIFD.UserComment: helper.UserComment.dump(f'SiWIM_PHOTO_TS:{ts}'),
                    piexif.ExifIFD.ImageUniqueID: metadata.get(META_VEHICLE, 'N/A')
                }
            }
        return piexif.dump(exif_dict)


def get_relevant_tags_from_xml(xml: Element, tags: Tuple[str, ...]) -> Dict[str, Any]:
    """ Finds relevant tags in XML.
    :param xml: Etree object containing the data.
    :param tags: List of tag names.
    :return: Dictionary with relevant tags and their values.
    """
    result = {}
    for tag in tags:
        result[tag] = xml.find(tag).text if xml.find(tag) is not None and xml.find(tag).text is not None else 'N/A'
    return result
