import datetime

import serial

from abstract.module import Module
from consts import MOD_THIES, SAVE_HEADER, SAVE_PATH, SAVE_SUFFIX, THIES_INTERVAL, THIES_PORT
from exceptions import StopModule

THIES_STX = '\x02'
THIES_ETX = '\x03'
THIES_DLM = ' '
THIES_CRCM = '*'
THIES_ENDLN = '\r'
THIES_CMD = '00TR4'
THIES_PARMS = [
    ('Wind speed', 'm/s', float),
    ('Wind direction', 'deg', int),
    ('Temperature', 'C', float),
    ('Relative humidity', '%', int),
    ('Air pressure', 'hPa', float),
    ('Max brightness', '', int),
    ('Direction of brightness', 'deg', int),
    ('Precipitation intensity', 'mm/h', float),
    ('Precipitation event', '0/1', int),
    ('Latitude', 'deg', float),
    ('Longitude', 'deg', float),
    ('Height', 'm', int),
    ('Sun elevation', 'deg', float),
    ('Sun azimuth', 'deg', float),
    ('Date', '%d.%m.%y', datetime.date),
    ('Time', '%H:%M:%S', datetime.time)
]


class ThiesModule(Module):
    def __init__(self, args):
        self.baud = 9600
        self.interval: int = 30
        Module.__init__(self, args, mandatory_keys=(THIES_PORT,), optional_keys=(THIES_INTERVAL,))
        self.type = MOD_THIES

    def thies_crcok(self, s) -> bool:
        if len(s) > 3:
            if s[-3] == THIES_CRCM:
                x = 0
                for c in s[:-3]:
                    x ^= ord(c)
                try:
                    return x == int(s[-2:], 16)
                except Exception as e:
                    self.logger.error(f'Error while parsing CRC: {e}')
                    return False
        return False

    def thies_request(self, ser, cmd) -> bool:
        ser.flushInput()
        ser.write(cmd.encode())
        ser.write(THIES_ENDLN.encode())
        s = ''
        while True:
            b = ser.read(32)

            if len(b) == 0:  # EOF
                break

            s = s + b.decode('ascii')

            if s.find(THIES_ETX) > -1:
                break

            if len(s) > 512:  # safeguard
                break

        si = s.find(THIES_STX)
        if si >= 0:
            ei = s.find(THIES_ETX, si)
            if ei > si:
                s = s[si + 1:ei].strip()
                if self.thies_crcok(s):
                    return s
                self.logger.error(f'Checksum error {s}.')
        return False

    def thies_convert(self, s) -> list:
        l = list(filter(None, s.split(THIES_DLM)))
        l.pop()  # remove csum

        for i in range(len(l)):
            if i < len(THIES_PARMS):
                t = THIES_PARMS[i][2]
                try:
                    if t == float:
                        l[i] = float(l[i])
                    elif t == int:
                        l[i] = int(l[i])
                    elif t == datetime.time:
                        l[i] = datetime.datetime.strptime(l[i], THIES_PARMS[i][1]).time()
                    elif t == datetime.date:
                        l[i] = datetime.datetime.strptime(l[i], THIES_PARMS[i][1]).date()
                except Exception as e:
                    self.logger.error(f'Problems converting {THIES_PARMS[i][0]} with value {l[i]}: {e}')
                    l[i] = None

        return l

    def thies_main(self, ser):
        r = self.thies_request(ser, THIES_CMD)
        if r:
            l = self.thies_convert(r)

            for i in range(len(l)):
                yield l[i]

    def run(self) -> None:
        self.alive = True
        self.end = False
        self.logger.debug('Started.')
        try:
            ser = serial.Serial(self.port, self.baud)
            ser.timeout = 1

            while True:
                self.throttle()
                if self.end:
                    raise StopModule('Shutdown flag detected.')

                data = list(self.thies_main(ser))

                header = '\t'.join(['ts'] + [f'{item[0]} ({item[1]})' for item in THIES_PARMS])
                self.write_data({SAVE_PATH: 'weather', SAVE_HEADER: header, SAVE_SUFFIX: 'tsv'}, [str(d) for d in data])
                self.zzzzz(self.interval)

        except StopModule as e:
            self.log_stop_module(e)
        finally:
            try:
                ser.close()
            except UnboundLocalError:  # Ignore, if ser was never initialized.
                pass
            self.logger.debug('Stopped.')
