#!/usr/bin/python3
import os
import platform
import sys
import threading
import time
from argparse import ArgumentParser
from datetime import datetime
from pathlib import Path

import tomlkit.exceptions
from cestel_helpers.console import configure_all
from cestel_helpers.exceptions import ConfError
from cestel_helpers.i_conf_manager import SEC_DEFAULT, get_sections, init_toml, read_conf, read_confs
from cestel_helpers.log import init_logger, remove_handlers
from cestel_helpers.update import is_update_available
from cestel_helpers.version import check_file_hashes, get_version, iter_incorrect_files
from packaging.version import InvalidVersion, parse

import config
from acquisition.binary.arh import CameraArh
from acquisition.binary.axis import CameraAxis
from acquisition.binary.cammra import CameraCammra
from acquisition.binary.mobotix import CameraMobotix
from acquisition.can import CanModule
from acquisition.efoy import EfoyModule
from acquisition.general.arh import AcquisitionArh
from acquisition.general.comark import AcquisitionComark
from acquisition.general.ffgroup import AcquisitionFfgroup
from acquisition.general.siwim import AcquisitionSiwim
from acquisition.general.tecdetect4 import AcquisitionTecdetect4
from acquisition.hysteresis import HistModule
from acquisition.papago import PapagoModule
from acquisition.thies import ThiesModule
from aggregation_module import AggregationModule
from communication_module import CommunicationModule
from consts import (COMM_BACKLOG, COMM_BUFFER, COMM_PORT, CONF_BACKUP_DIR, CONF_DIR, CONF_DOWNSTREAM, CONF_GLOBAL, GLOBAL_AUTO_UPDATE, GLOBAL_UPDATE_CHANNEL, HASH_FILE, LOGGER_MAIN, LOG_DIR, LOG_LEVEL, LOG_LEVEL_CON, MOD_AGGR, MOD_CAN, MOD_COMM,
                    MOD_EFOY, MOD_EVENTS, MOD_GAP, MOD_HYST, MOD_MATCH, MOD_NAME, MOD_OCLI, MOD_OUT, MOD_PAP, MOD_PIC, MOD_STAT, MOD_THIES, ROOT_OUTPUT)
from exceptions import RestartI, ShutdownI, StopModule
from helpers.helpers import exit_with_prompt, get_current_site
from output.events import OutputEventsModule
from output.output_client_module import OutputClientModule
from output.output_module import OutputModule
from photo_servers.siwim import PhotoSiwim
from postprocessing.gap import GapModule
from postprocessing.match import MatchModule
from status_module import StatusModule

MODULES = {  # Holds mappings from module names in I to tuple of object type and object parameters
    MOD_AGGR: AggregationModule,
    MOD_CAN: CanModule,
    MOD_COMM: CommunicationModule,
    MOD_EFOY: EfoyModule,
    MOD_GAP: GapModule,
    MOD_HYST: HistModule,
    MOD_MATCH: MatchModule,
    MOD_OUT: OutputModule,
    MOD_PAP: PapagoModule,
    MOD_PIC: PhotoSiwim,
    MOD_STAT: StatusModule,
    MOD_OCLI: OutputClientModule,
    MOD_EVENTS: OutputEventsModule,
    MOD_THIES: ThiesModule
}

REC_MODULES = {
    'arh': AcquisitionArh,
    'comark': AcquisitionComark,
    'ffgroup': AcquisitionFfgroup,
    'siwim': AcquisitionSiwim,
    'tecdetect4': AcquisitionTecdetect4
}

CAM_MODULES = {
    'arh': CameraArh,
    'axis': CameraAxis,
    'ffgroup': CameraCammra,
    'mobotix': CameraMobotix
}

OUT_MODULES = {
    'xml_server': OutputModule,
    'xml_client': OutputClientModule,
    'events_server': OutputEventsModule
}
POS_MODULES = {
    'match': MatchModule,
    'gap': GapModule
}

if __name__ == '__main__':
    # Get the script parameters
    parser = ArgumentParser()
    parser.add_argument('--init_site', action='store_true', help='pass if basic configuration should be created')
    parser.add_argument('--sites_dir', type=str, help='pass, if you want files to be written to a different folder')
    parser.add_argument('--siwim_e_conf', default=config.siwim_e_conf, type=str, help='siwim_e_conf that should be used to determine site name')
    parser.add_argument('--siwim_i_dir', type=str, help='pass if I should write configuration and log files to a different folder')
    args, launcher_args = parser.parse_known_args()

    # This logger handles essentials of the application and is relevant until configuration files can be read.
    initial_logger = init_logger(LOGGER_MAIN, log_to='siwim_i', console_level=10, to_console=True, to_file=False)

    if os.path.dirname(sys.argv[0]) != '':
        os.chdir(os.path.dirname(sys.argv[0]))

    # Modify location of confs and logs.
    if args.siwim_i_dir:
        config.conf_dir = Path(args.siwim_i_dir, CONF_DIR)
        config.log_dir = Path(args.siwim_i_dir, LOG_DIR)

    # Obtain site name.
    try:
        config.site_name = get_current_site(args.siwim_e_conf, set_i_dir=False if args.siwim_i_dir is not None else True)  # If siwim_i_dir was passed, we don't want get_current_site to update i_conf and i_log directories.
    except ConfError as e:
        initial_logger.critical(f'Could not read SiWIM-E configuration file: {e}')
        exit_with_prompt()

    # Modify sites directory.
    if args.sites_dir:
        config.sites_dir = Path(args.sites_dir)
        config.conf_dir = config.sites_dir / config.site_name / CONF_DIR
        config.conf_backup_dir = config.sites_dir / config.site_name / CONF_BACKUP_DIR
        config.log_dir = config.sites_dir / config.site_name / LOG_DIR

    # Create blank configuration.
    if args.init_site:
        import cestel_helpers.i_conf_manager as i_conf

        # Generate global.toml.
        if not Path(config.conf_dir, CONF_GLOBAL).exists():
            global_conf = i_conf.init_toml(config.site_name, default_values=i_conf.defaults[i_conf.CONF_GLOBAL])
            i_conf.write_conf(global_conf, config.conf_dir / CONF_GLOBAL, sort=True)
        else:
            initial_logger.error(f'Conf {CONF_GLOBAL} initialization failed because the conf already exists.')
        # Generate downstream.toml.
        if not Path(config.conf_dir, CONF_DOWNSTREAM).exists():
            downstream_conf = i_conf.init_toml(config.site_name)
            i_conf.write_conf(downstream_conf, config.conf_dir / CONF_DOWNSTREAM)
        else:
            initial_logger.error(f'Conf {CONF_DOWNSTREAM} initialization failed because the conf already exists.')

    try:
        conf_global = read_conf(Path(config.conf_dir, CONF_GLOBAL))[SEC_DEFAULT]
    except (ConfError, tomlkit.exceptions.NonExistentKey) as e:  # TODO Remove NonExistentKey once it's added to cestel-helpers.
        initial_logger.critical(f'Invalid conf at {config.conf_dir}: {e} If you\'re trying to initialize a site, use --init_site.')
        exit_with_prompt()
    try:
        downstream = read_conf(Path(config.conf_dir, CONF_DOWNSTREAM))[SEC_DEFAULT]
    except (ConfError, tomlkit.exceptions.NonExistentKey) as e:  # TODO Remove NonExistentKey once it's added to cestel-helpers.
        downstream = init_toml(config.site_name)

    config.auto_update = conf_global.get(GLOBAL_AUTO_UPDATE)  # noqa
    config.root_output = conf_global.get(ROOT_OUTPUT)
    with config.lock:
        config.status_dict[GLOBAL_UPDATE_CHANNEL] = conf_global.get(GLOBAL_UPDATE_CHANNEL)

    remove_handlers(initial_logger)
    del initial_logger  # Initial logger is no longer useful.
    logger = init_logger(LOGGER_MAIN, folder=config.log_dir, log_to='siwim_i', level=conf_global.get(LOG_LEVEL, 20), console_level=conf_global.get(LOG_LEVEL_CON, 20), to_console=bool(conf_global.get(LOG_LEVEL_CON, True)),
                         add_line_number=True if conf_global.get(LOG_LEVEL_CON) == 10 else False, no_date=True)

    restart = True if platform.system() == 'Windows' else False  # This flag is used to determine whether application should restart or not when receiving shutdown signal.

    version = get_version()
    if Path(HASH_FILE).exists() and not check_file_hashes():
        logger.warning('Hashes don\'t match! Displayed version may be incorrect.')
        logger.info(f'Mismatched hashes: {[f for f in iter_incorrect_files()]}')

    APP_NAME = f'SiWIM-I v{version} Core'

    configure_all(app_name=APP_NAME)

    logger.info(f'{APP_NAME} starting for {config.site_name}. Using {config.status_dict[GLOBAL_UPDATE_CHANNEL]} update channel.')
    logger.debug('Running in debug mode.')

    # Notify user when a non-standard SiWIM-E configuration file was set.
    if args.siwim_e_conf != config.siwim_e_conf:
        logger.info(f'Using SiWIM-E configuration file located at {args.siwim_e_conf}.')
        config.siwim_e_conf = args.siwim_e_conf

    # Read configuration files for all modules.
    try:
        conf_modules = read_confs(config.conf_dir)
    except ConfError as e:
        logger.critical(e)
        exit_with_prompt()

    if not config.sites_dir.exists():
        logger.critical(f'Sites directory doesn\'t exist at {config.sites_dir}. Exiting ...')
        exit_with_prompt()

    logger.debug(f'Sites location is {config.sites_dir}. Configuration files loaded from {config.conf_dir}. Logging done to {config.log_dir}')

    if not config.auto_update:
        logger.warning('Automatic updates are disabled! Neither libraries nor the application will update.')

    running = {}
    running_threads = {}  # Meant to replace "running". Instead of checking if Module classes are running, we'll check if threads are running.

    comm_started = False
    acq_exists = False

    # factory
    for name, module_type_conf in conf_modules.items():  # noqa  # Iterate over configuration for module types.
        for section in get_sections(module_type_conf):  # Iterate over sections (skips DEFAULT).
            if name == MOD_COMM:  # If a communication module was defined in conf file, we don't have to start it later.
                comm_started = True
                comm_module_name = section
            params = {}
            for key, val in module_type_conf[section].items():
                params[key] = val

            params[MOD_NAME] = section
            typ = None
            try:
                if name == 'acquisition':  # Module is a virtual class further split into multiple modules
                    typ = params['type']
                    if typ not in REC_MODULES:  # It would be prettier with try ... except, but we want any KeyErrors that happen in the function to come through rather than be caught here.
                        logger.critical(f'{section} failed: {name}:{typ} not recognized.')
                        continue
                    acq_exists = True
                    running[section] = REC_MODULES[typ](params)
                elif name == 'camera':
                    typ = params['type']
                    if typ not in CAM_MODULES:  # It would be prettier with try ... except, but we want any KeyErrors that happen in the function to come through rather than be caught here.
                        logger.critical(f'{section} failed: {name}:{typ} not recognized.')
                        continue
                    running[section] = CAM_MODULES[typ](params)
                elif name == 'output':
                    typ = params['type']
                    if typ not in OUT_MODULES:
                        logger.critical(f'{section} failed: {name}:{typ} not recognized.')
                        continue
                    running[section] = OUT_MODULES[typ](params)
                elif name == 'postprocessing':
                    typ = params['type']
                    if typ not in POS_MODULES:
                        logger.critical(f'{section} failed: {name}:{typ} not recognized.')
                        continue
                    running[section] = POS_MODULES[typ](params)
                else:
                    try:
                        running[section] = MODULES[name](params)  # Create instance of MODULES[name] with parameters read from configuration file.
                    except KeyError:
                        raise StopModule(f'{name} is not a supported module type. {name}.toml ignored.')
            except StopModule as e:
                logger.critical(f'{section} failed: {e}')

            if section not in downstream:  # noqa
                downstream[section] = []  # TODO This may not be necessary; it was done for simplicity.

    if not acq_exists:
        logger.error('No acquisition module defined. There is no way for the application to obtain data!')

    # Add the communication module, if it hasn't been defined (usual use case).
    if not comm_started:
        name = "comm_default"
        comm_module_name = name
        downstream[name] = []
        running[name] = CommunicationModule({MOD_NAME: name, COMM_BACKLOG: 5, COMM_BUFFER: 2048, COMM_PORT: 8173})

    for key, module in running.items():
        if module.restart_on_fail:
            logger.info(f'Starting {key} ...')
            for downster in downstream[key]:
                try:
                    module.add_downstream_module(running[downster])
                except:
                    pass
            running_threads[key] = threading.Thread(target=module.run)
            running_threads[key].start()
        else:
            logger.critical(f'Module {module.name} not started. Check its log for details.')

    day = datetime.now().date()
    try:
        while True:
            try:
                for key, module in running.items():
                    if module.restart_on_fail and (not module.is_alive() or not running_threads[key].is_alive()):
                        try:
                            logger.info(f'Restarting {key}: {module.stop_reason}')
                            running_threads[key] = threading.Thread(target=module.run)
                            running_threads[key].start()
                        except:
                            logger.exception('Got exception while trying to restart a module:')
                loc = ""
                if config.auto_update and day != datetime.now().date():
                    day = datetime.now().date()
                    # TODO This should be addressed once there are no versions of SiWIM-I older than 5.0.0. It's purpose is to parse old devel versions in accordance with PEP440.
                    try:
                        PEP440_VERSION = str(parse(version))
                    except InvalidVersion:
                        logger.info('Resetting I due to invalid version ...')
                        raise SystemExit
                    if is_update_available('siwim-i', config.status_dict[GLOBAL_UPDATE_CHANNEL], PEP440_VERSION):
                        logger.info('Resetting I due to update ...')
                        raise SystemExit
                try:
                    running[comm_module_name].are_we_resetting_1()  # noqa
                except RestartI:
                    logger.info("Resetting I due to a frontend message ...")
                    raise SystemExit
                else:
                    time.sleep(1)
            except KeyboardInterrupt:
                raise ShutdownI
            except SystemExit:
                raise SystemExit
            except:
                logger.exception('Unexpected error:')
    except SystemExit:
        logger.info(f'{APP_NAME} exiting normally.')
    except ShutdownI:  # If this exception is raised, it means that the application should shut down rather than restart. Generally means a KeyboardInterrupt was caught.
        logger.warning('Shutdown signal received. Application will not restart.')
        restart = False
    finally:
        logger.info('Cleaning up ...')
        for key1, module1 in running.items():
            module1.set_end()

        shutdown_time = time.time()  # This timer is used to forcefully terminate the application.
        while True:
            all_dead = True
            for key1, module1 in running_threads.items():
                if module1.is_alive():
                    # TODO This is a workaround, since Gap and Match modules should never be interrupted. A better solution needs to be found ... maybe add a list?
                    if not (isinstance(running.get(key1), GapModule) or isinstance(running.get(key1), MatchModule)) and time.time() > shutdown_time + 15:
                        logger.error(f'{key1} did not shut down in time. Forcefully terminating.')
                        break
                    logger.debug(f'Waiting for {key1} to die ...')
                    all_dead = False
                    break
            if all_dead:
                logger.info('All threads have closed.')
                break
            time.sleep(0.5)
        if restart:  # If restart flag is set, we want to replace this application with a new instance. Additionally, if automatic updates are enabled, we run siwim_i.py; otherwise this script is simply restarted.
            os.execv(sys.executable, [sys.executable, 'siwim_i.py' if config.auto_update else 'core.py'] + [arg for arg in sys.argv if arg != 'core.py'])
