# Implementation of data acquisition from vidar cameras.
import base64
import copy
import datetime
import http.client
import time
from pathlib import Path

import requests
import requests.exceptions
from lxml import etree

from abstract.module import Module
from consts import MOD_TYPE, RCV_HOST, RCV_OFFSET, RCV_PASS, RCV_PORT, RCV_SAVE, RCV_USER, SAVE_PATH, SAVE_SUFFIX, TS_FORMAT_STRING
from exceptions import StopModule


class AcquisitionArh(Module):
    def __init__(self, args) -> None:
        # Define defaults that may be overwritten by the conf.
        self.mode = 'default'
        self.parser = etree.XMLParser(remove_blank_text=True)
        self.uname = None
        self.password = None

        # Initialize the module, passing it a list of keys that must exist for it to work properly.
        Module.__init__(self, args, mandatory_keys=(MOD_TYPE, RCV_HOST, RCV_OFFSET, RCV_PASS, RCV_PORT, RCV_SAVE, RCV_USER))
        self.type = 'arh'

    def parse_data(self, resp_text: str):
        try:
            resp_text = resp_text.replace('<?xml version="1.0" encoding="UTF-8"?>', "")
            parser = etree.XMLParser(remove_blank_text=True)
            xml_resp = etree.fromstring(resp_text, parser=parser)
            id_num = xml_resp.find("ID").attrib["value"]
            ts = xml_resp.find("capture").find("frametime").attrib["value"]
            ts = datetime.datetime.strptime(ts, "%Y.%m.%d %H:%M:%S.%f").strftime('%Y-%m-%d-%H-%M-%S-%f')[:-3]
            anpr = xml_resp.find("anpr").find("text").attrib["value"]
            speed = xml_resp.xpath('trigger/speed/@value')  # This will be either value of speed or empty list if it doesn't exist.
            lprimage = ""
            # try finding photo
            try:
                images = xml_resp.find("images")
                lprimage = images.find("lp_img").attrib["value"]
                # we don't want to save it
                images.getparent().remove(images)
            except:
                pass
            try:
                data = xml_resp.xpath('trigger/zone_name/@value')
                lane = data[0][1:]
            except IndexError:
                self.logger.error(f'Could not extract lane from trigger/zone_name/@value with value {data}.')
                return {}
            if self.save_original == 1:
                xml_resp.tag = 'vehicle'
                self.write_data({SAVE_PATH: 'anpr', SAVE_SUFFIX: 'xmll'}, [etree.tostring(xml_resp, encoding='utf-8', pretty_print=False).decode()])
            result = {'id': id_num, 'ts': ts, 'lp': anpr, 'lane': str(lane), 'lprimage': lprimage}
            if speed:  # Only add speed if list isn't empty.
                result['v'] = str(round(float(speed[0]) / 3.6, 2))
            return result
        except:
            self.logger.exception('Failed to parse Vidar response:')
            return {}

    def run(self) -> None:
        self.alive = True
        self.end = False

        try:
            if self.mode == 'default':  # This defines what we want to obtain from the camera.
                if not self.uname or not self.passw:
                    raise StopModule(msg='ARH credentials not set. Returning.')
                login_url = "http://" + self.host + ":" + str(self.port) + "/login.html?p_send=1&p_username=" + self.uname + "&p_passw=" + self.passw
            # WHERE ID> should be followed by previous ID
            result_list_query_url = "http://" + self.host + ":" + str(self.port) + "/lpr/cff?cmd=getresultlist&select=WHERE ID>"
            # first iteration follow id= with last, then with IDs from result list query
            dl_url = "http://" + self.host + ":" + str(self.port) + "/lpr/cff?cmd=getdata&id="
            id_num = "last"
            with requests.Session() as s:
                while True:
                    self.throttle()
                    resp_dicts = []
                    if self.mode == 'default':
                        try:
                            s.get(login_url, timeout=5)
                        except Exception as e:
                            raise StopModule(msg=f'Login timeout: {e}')
                        if id_num == "last":
                            try:
                                resp = s.get(dl_url + id_num, timeout=2)
                            except requests.exceptions.ReadTimeout:
                                raise StopModule(msg=f'Read timeout reached for request: {dl_url + id_num}')
                            except Exception as e:
                                raise StopModule(msg=f'Data acquisition timeout: {e}')
                            resp_dict = self.parse_data(resp.text)
                            if len(resp_dict) == 0:
                                self.logger.error('No data in response!')
                                self.logger_dump.error(resp.text)
                                self.zzzzz(2)
                                continue
                            id_num = resp_dict["id"]
                            self.logger.debug(f'Adding {resp_dict["id"]} at {resp_dict["ts"]} (lane {resp_dict["lane"]}) with plate: {resp_dict["lp"].encode()}')
                            resp_dicts.append(resp_dict)
                        else:
                            # get all new event ids
                            try:
                                resp1 = s.get(result_list_query_url + id_num, timeout=3)
                            except requests.exceptions.ReadTimeout:
                                raise StopModule(f'Read timeout reached for request: {result_list_query_url + id_num}')
                            except http.client.RemoteDisconnected:
                                raise StopModule(f'Camera did not respond for request: {result_list_query_url + id_num}')
                            except Exception as e:
                                raise StopModule(f'Result list acquisition timeout: {e}')
                            xml_resp = etree.fromstring(resp1.text)
                            ids = []
                            for child in xml_resp.getchildren():
                                if child.tag.startswith("result"):
                                    ids.append(child.attrib["value"])
                            ids = sorted(ids)
                            for oneid in ids:
                                id_num = oneid
                                try:
                                    resp2 = s.get(dl_url + id_num, timeout=2)
                                except (requests.exceptions.ConnectionError, requests.exceptions.ReadTimeout):
                                    raise StopModule(msg=f'Read timeout reached or connection aborted for request: {dl_url + id_num}')
                                except Exception as e:
                                    raise StopModule(msg=f'Data acquisition timeout: {e}')
                                resp_dict = self.parse_data(resp2.text)
                                if len(resp_dict) == 0:
                                    continue
                                self.logger.debug(f'Adding {resp_dict["id"]} at {resp_dict["ts"]} (lane {resp_dict["lane"]}) with plate: {resp_dict["lp"].encode()}')
                                resp_dicts.append(resp_dict)
                    elif self.mode == 'axlecount':  # This implementation only saves data to disk.
                        if id_num == 'last':
                            query_list = f'http://{self.host}:{self.port}/lpr/cff?cmd=getid&id=last'
                            id_xpath = '/ans/id/@value'
                        else:
                            query_list = f'http://{self.host}:{self.port}/lpr/cff?cmd=getresultlist&select=WHERE ID>{id_num}'
                            id_xpath = '//results/*[starts-with(name(), "result_")]/@value'
                        try:
                            resp = s.get(query_list, timeout=2)
                        except (requests.exceptions.ReadTimeout, requests.exceptions.ConnectTimeout) as e:
                            raise StopModule(f'Camera did not respond in time: {e}')
                        id_list = etree.fromstring(resp.text).xpath(id_xpath)
                        try:
                            id_num = id_list[0]
                        except IndexError:
                            if resp.text.find('SQL error - \'not an error') != -1:  # This is the camera's response when there is no data to send.
                                self.logger.debug('Camera responded that there\'s no data.')
                            else:
                                self.logger.debug(f'Could not extract an id list from {resp.text}.')
                            time.sleep(1)
                            continue

                        query_result = f'http://{self.host}:{self.port}/lpr/cff?cmd=getdata&id='

                        self.logger.info(f'List of IDs to process: {id_list}.')
                        for id in reversed(id_list):  # IDs are returned in reversed order.
                            try:
                                resp = s.get(query_result + id)
                            except requests.exceptions.ConnectionError as e:
                                raise StopModule(f'An error has occurred while requesting data: {e}')
                            xml = etree.fromstring(resp.content, parser=self.parser)
                            # Save the image.
                            if xml.tag == 'result':
                                ts = datetime.datetime.fromtimestamp(int(xml.xpath('/result/capture/frametimems/@value')[0]) / 1000).strftime(TS_FORMAT_STRING)[:-3]
                            elif xml.tag == 'ans':
                                error_code = xml.xpath('/ans/ecode/@value')[0]
                                error_msg = xml.xpath('/ans/estr/@value')[0]
                                self.logger.error(f'Server responded with code {error_code}: {error_msg}.')
                                continue
                            else:
                                self.logger.error(f'Unrecognized response. See error.txt for details.')
                                with open(Path(self.data_dirs[self.mode], f'error.txt'), 'ab') as fd:
                                    fd.write(resp.content + b'\n')
                                continue

                            img_dir = self.generate_data_dir(self.mode, ts=ts, update=False)
                            base64_img_list = xml.xpath('/result/images/aux_img/@value')
                            for i, img in enumerate(base64_img_list):  # Save all the images.
                                if img != 'n.a.':  # Only continue if the data actually exists.
                                    image_blob = base64.b64decode(img)
                                    with open(img_dir / f'{ts}_{i}.jpg', 'wb') as img_file:
                                        img_file.write(image_blob)

                            if self.save_original:
                                images = xml.find('images')
                                if images is not None:
                                    xml.remove(images)
                                self.write_data({SAVE_PATH: 'axlecount'}, etree.tostring(xml, encoding='utf-8').decode())
                            resp_dicts.append(xml)

                    if self.mode == 'default':
                        for resp_dict in resp_dicts:
                            vehicle_node = etree.Element("vehicle")
                            anpr_el = etree.Element("anpr")
                            anpr_el.set('source', self.name)
                            anpr_el.set('type', 'arh')
                            for key in resp_dict.keys():
                                key_el = etree.Element(key)
                                key_el.text = resp_dict[key]
                                anpr_el.append(key_el)
                            vehicle_node.append(anpr_el)
                            # print(etree.tostring(vehicle_node))
                            for key, mod in self.downstream_modules_dict.items():
                                mod.add_vehicle(copy.deepcopy(vehicle_node), self.name)
                            self.logger.info(f'Sent {resp_dict.get("ts")}_{resp_dict.get("lane")} downstream.')

        except StopModule as e:
            self.log_stop_module(e)
