import re
import socket
import time

from abstract.module import Module
from consts import EFOY_BUFFER, EFOY_HOST, EFOY_INTERVAL, EFOY_PORT, EFOY_SAVE, MOD_EFOY, SAVE_HEADER, SAVE_PATH, SAVE_SUFFIX
from exceptions import StopModule


class EfoyModule(Module):
    def __init__(self, args):
        self.interval: int = 3600
        self.buffer: int = 2048
        Module.__init__(self, args, mandatory_keys=(EFOY_HOST, EFOY_PORT, EFOY_SAVE), optional_keys=(EFOY_BUFFER, EFOY_INTERVAL))
        self.type = MOD_EFOY

    def run(self):
        self.alive = True
        self.end = False
        try:
            sent = False
            received_at = time.time()
            while True:  # Main loop that's saving fuel level.
                self.throttle()
                if self.end:
                    self.alive = False
                    raise StopModule('Shutdown flag detected.')
                line = ''
                fuel_level = ''
                # Only do this once per polling_interval.
                if not sent or time.time() - received_at > self.interval:
                    if not self.connect_tcp():
                        self.alive = False
                        raise StopModule('Connection failed.')

                    self.logger.debug('Sending cartridge request ...')
                    self.s.send(b'cartridge\r')
                    sent = True
                    try:
                        # Attempt to obtain data from edgar.
                        retries = 0
                        while retries < 15:  # Obtain the data.
                            received = self.s.recv(self.buffer).decode()
                            self.logger.debug(f'Received response: {received}')
                            if received == '':
                                self.alive = False
                                raise StopModule('Received an empty string, exiting thread.')
                            line = line + received.replace('\r', str())  # We have to remove \r for parsing in next step to work. TODO Refactor and optimize this.
                            self.logger.info(f'Response so far: {line}')
                            match = re.search(r'M28 \((\d+)%\)', line)
                            if match:
                                fuel_level = match.group(1)
                                received_at = time.time()
                                break
                            else:  # Full result hasn't been received yet.
                                retries += 1
                                continue
                        else:
                            self.alive = False
                            raise StopModule(f'Unexpected response: {line}')
                    except socket.timeout:
                        self.logger.warning('Connection timed out. Retrying ...')
                        time.sleep(60)
                        continue
                    except socket.error:
                        self.logger.debug(f'{self.s} with buffer [{self.buffer}] closed.')
                        self.alive = False
                        raise StopModule('Connection closed.')
                    finally:
                        self.logger.debug('Closing socket.')
                        self.s.close()
                    if self.save_to_disk:
                        self.write_data({SAVE_PATH: 'fuel', SAVE_HEADER: 'ts\tfuel_level[%]', SAVE_SUFFIX: 'tsv'}, fuel_level)
                    self.set_upstream_info(self.get_name(), "fuel_level", fuel_level)
                time.sleep(1)
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
        self.alive = False
