from abc import ABC
from pathlib import Path
from typing import Any, Dict, List, Set, Tuple

from cestel_helpers.aliases import Element
from lxml import etree

import config
from abstract.module import Module
from consts import POST_FUZZINESS, POST_GRACE, POST_LOOKBACK, POST_SWD_PATH


class PostprocessingModule(Module, ABC):
    def __init__(self, module_data: Dict[str, Any], mandatory_keys: Tuple[str, ...] = tuple(), optional_keys: Tuple[str, ...] = tuple()):
        """ Each module is initialized from a dictionary of values read from conf.
        :param module_data: Dictionary containing everything defined in configuration file of the module.
        :param mandatory_keys: A tuple of keys that must exist in the toml file for the module to work correctly.
        :param optional_keys: A tuple of keys that are recognized by the module, but don't need to be defined in the toml file.
        """
        self.lookback_days: int = 5
        self.grace_period: int = 1800
        self.swd_dir = config.sites_dir / config.site_name / 'ext'
        Module.__init__(self, module_data, mandatory_keys=mandatory_keys, optional_keys=optional_keys + (POST_FUZZINESS, POST_GRACE, POST_LOOKBACK, POST_SWD_PATH))
        self.swd_dir = Path(self.swd_dir)  # Make sure it's a Path object, in case it got set via toml.

    def save_swd_file(self, file_name: str, vehicles: List[Element], duplicates: Set[Element]) -> None:
        path = self.swd_dir / file_name
        if path.is_file():  # Make sure the file is preserved if something goes wrong.
            with open(path, encoding='utf-8') as f:
                backup_data = f.read()
        try:
            with open(path, 'w', encoding='utf-8') as output_file:
                # Write header.
                output_file.write(f'<?xml version="1.0" ?>\n<swd version="1"><site><name>{config.site_name}</name><vehicles>\n')
                # Write vehicles with gaps.
                for vehicle in [v for v in vehicles if v not in duplicates]:
                    output_file.write(etree.tostring(vehicle).decode().strip() + '\n')
                # Write footer.
                output_file.write('</vehicles></site></swd>')
        except Exception as e:  # If anything happens during writing, write the original file back.
            with open(path, 'w', encoding='utf-8') as output_file:
                output_file.write(backup_data)
            raise

    def swd_to_process(self, xml_file: Path) -> bool:
        """ Checks if the file needs to be processed.
        """
        try:
            processed = len(etree.parse(str(xml_file)).getroot().xpath(f'/swd/site/vehicles/vehicle/integration/{self.type}')) != 0
            self.logger.info(f'{xml_file.name} {"does not need" if processed else "needs"} to be updated.', stacklevel=2)
            return not processed
        except OSError:
            self.logger.warning(f'Could not find file for {xml_file.name}.', stacklevel=2)
            return False
        except etree.XMLSyntaxError:
            self.logger.critical(f'{xml_file.name} is corrupted!', stacklevel=2)
            return False
