#!/usr/bin/python3
# Martin Konečnik, http://git.siwim.si/cestel/siwim-i
# Script serves as a launcher to SiWIM-I v5. It takes care of updating, installing missing libraries and restarting core.py (formerly known as siwim_i.py).
import os
import shutil
import subprocess
import sys
from argparse import ArgumentParser
from logging import getLogger
from pathlib import Path
from zipfile import ZipFile

import config
from consts import CONF_DIR, CONF_GLOBAL, GLOBAL_ALLOW_ROLLBACK, GLOBAL_UPDATE_CHANNEL, LOGGER_MAIN, LOG_DIR


def update_libraries() -> bool:
    try:
        getLogger().info('Updating libraries using uv ...')
        subprocess.run(f'uv sync', check=True, shell=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        return True
    except subprocess.CalledProcessError as e:
        getLogger().debug(f'Output of command: {e.stdout}')
        return False


if __name__ == '__main__':
    parser = ArgumentParser()
    parser.add_argument('--disable_core', action='store_true', help='application will not replace itself with core.py on finish')
    parser.add_argument('--init_site', action='store_true', help='pass if basic configuration should be created')
    parser.add_argument('--sites_dir', type=str, help='pass, if you want files to be written to a different folder')
    parser.add_argument('--siwim_e_conf', default=config.siwim_e_conf, type=str, help='siwim_e_conf that should be used to determine site name')
    parser.add_argument('--siwim_i_dir', type=str, help='pass if I should write configuration and log files to a different folder')
    parser.add_argument('-v', '--verbose', action='count', default=0, help='run with debug output')
    args, core_args = parser.parse_known_args()

    libraries_ok = update_libraries()

    if not libraries_ok:
        getLogger().critical(f'Libraries could not be updated. Run uv sync and open an issue with the error message.')

    import requests
    from cestel_helpers.log import init_logger
    from cestel_helpers.version import get_version
    from cestel_helpers.update import download_update
    from cestel_helpers.exceptions import ConfError, UpdateError
    from cestel_helpers.i_conf_manager import SEC_DEFAULT, read_conf
    from helpers.helpers import exit_with_prompt, get_current_site
    from packaging.version import parse

    # Obtain site name and define conf and log directories
    config.site_name = get_current_site(args.siwim_e_conf)
    if args.siwim_i_dir:
        config.conf_dir = Path(args.siwim_i_dir, CONF_DIR)
        config.log_dir = Path(args.siwim_i_dir, LOG_DIR)

    logger = init_logger(LOGGER_MAIN, folder=config.log_dir, log_to='launcher', level=20 - args.verbose * 10, console_level=20 - args.verbose * 10, to_console=True, no_date=True, add_line_number=True if args.verbose else False)
    FULL_VERSION = get_version()
    PEP440_VERSION = str(parse(FULL_VERSION))
    APP_NAME = f'SiWIM-I v{FULL_VERSION} Launcher'
    logger.info(f'{APP_NAME} starting.')
    if core_args and core_args != ['']:
        logger.info(f'Arguments {core_args} not parsed and will be forwarded to core.py!')

    try:
        conf_global = read_conf(Path(config.conf_dir, CONF_GLOBAL))[SEC_DEFAULT]
    except ConfError as e:
        logger.critical(f'Invalid conf at {config.conf_dir}: {e}')
        exit_with_prompt()
    channel = conf_global.get(GLOBAL_UPDATE_CHANNEL)
    rollback = conf_global.get(GLOBAL_ALLOW_ROLLBACK)

    try:
        logger.debug(f'Update channel: {channel}')
        if channel:  # Blank channel isn't considered a valid channel.
            if download_update('siwim-i', channel, 'siwim_i.zip', PEP440_VERSION, execute=False, allow_rollback=rollback):
                logger.info('Latest version downloaded successfully. Unpacking ...')
                with ZipFile('siwim_i.zip') as zf:
                    current_feature_ver = int(PEP440_VERSION.split('.')[1])
                    new_feature_ver = int(zf.read('.version').decode().split('.')[1])
                    if current_feature_ver != new_feature_ver:  # If feature version changed, delete everything before update.
                        logger.info(f'Feature version changed ... Cleaning up {Path(".").absolute()}')
                        for p in sorted(Path('.').rglob('*'), reverse=True):
                            if p.name == 'siwim_i.zip' or '.venv' in p.parts:  # Delete everything except downloaded file and virtual environment.
                                continue
                            if p.is_file():
                                p.unlink()
                            elif p.is_dir():
                                if p.is_symlink():
                                    p.unlink()
                                else:
                                    shutil.rmtree(p)
                    zf.extractall('.')
                os.remove('siwim_i.zip')
                logger.info('Updating libraries ...')
                libraries_ok = update_libraries()
                if not libraries_ok:
                    logger.critical('Failed to update libraries. Application may not function correctly.')
            else:
                logger.debug(f'No update available for channel {channel}.')
    except UpdateError as e:
        logger.warning(f'Updater exited with error: {e}')
    except (requests.exceptions.ConnectTimeout, requests.exceptions.ConnectionError):
        logger.warning('Update failed, because application could not connect to update host. If this problem persists, make sure a valid DNS is set and software server is reachable.')
    except:
        logger.exception(f'Updating exited with an unexpected error for channel {channel}.')

    if not args.disable_core:
        # When run from core.py, the program name will be passed, and we don't want duplicates.
        new_args = [arg for arg in sys.argv if arg != 'siwim_i.py']
        getLogger().debug(f'core.py started with arguments {new_args}')
        os.execv(sys.executable, [sys.executable, 'core.py'] + new_args)
