import datetime
import platform
import socket
import struct
import threading
import time
from pathlib import Path

from lxml import etree

import config
from abstract.module import Module
from consts import OUT_FILTER_RULES, OUT_PORT, SAVE_PATH, TS_FORMAT_STRING
from exceptions import StopModule

if platform.system() == 'Windows':
    import pywintypes
    import win32con
    import win32file

class OutputEventsModule(Module):
    filter_rules: list
    heartbeat_interval: int
    max_clients: int
    port: int
    save_data: bool

    def __init__(self, args):
        # Process filter_rules
        if OUT_FILTER_RULES in args:
            args[OUT_FILTER_RULES] = [rule.split(',') for rule in args[OUT_FILTER_RULES]]
        Module.__init__(self, args, mandatory_keys=(OUT_PORT,))
        self.type = 'siwim_events'
        self.parser = etree.XMLParser(remove_blank_text=True)
        # client-thread indice, vehicle deque pairs
        self.vehicles_dict = {}
        self.clients_lock = threading.Lock()
        self.s = None

    def set_end(self):
        self.end = True
        self.s.close()  # This will close the socket, causing accept() call to fail with OSError.

    def find_lowest_unused_id(self):
        n = 0
        with self.clients_lock:
            while True:
                if n not in self.vehicles_dict:
                    return n
                n += 1

    def clientthread_xml(self, conn, idx):
        try:
            last_send_at = datetime.datetime.now()
            while True:
                self.throttle()
                if self.end:
                    try:
                        conn.shutdown(socket.SHUT_WR)
                    except OSError as e:
                        self.logger.info(f'Client: {e}')
                    conn.close()
                    self.logger.info(f'Client ended.')
                    return
                if len(self.vehicles_dict[idx]) > 0:
                    try:
                        with self.clients_lock:
                            vehicle = self.vehicles_dict[idx].popleft()
                        complete_xml = '<swd version="1"><site><name>' + config.site_name + '</name><vehicles>' + etree.tostring(vehicle).decode() + '</vehicles></site></swd>'
                        to_send = etree.fromstring(complete_xml, self.parser)
                    except:
                        self.logger.exception('Failed to form a string from xml:')
                        time.sleep(0.5)
                        continue
                    data = vehicle.find('wim/ts').text if vehicle.find('wim/ts') is not None else 'unknown'
                    self.logger.debug(f'Sending {data} to {idx}')
                    conn.send(etree.tostring(to_send, pretty_print=False))
                    last_send_at = datetime.datetime.now()
                if self.heartbeat_interval != 0 and (datetime.datetime.now() - last_send_at).total_seconds() > self.heartbeat_interval:
                    conn.send(self.form_message({'ts': datetime.datetime.now().strftime(TS_FORMAT_STRING)[:-3]}))
                    last_send_at = datetime.datetime.now()
                time.sleep(0.5)
        except (BrokenPipeError, ConnectionAbortedError, ConnectionResetError) as e:
            self.logger.info(f'Client {idx} closed: {e}')
        except:
            self.logger.exception(f'Client {idx} terminated:')
        finally:
            try:
                with self.clients_lock:
                    del self.vehicles_dict[idx]
                    self.logger.debug(f'Client {idx} removed from active thread list.')
                conn.close()
            except KeyError:
                self.logger.warning(f'Client {idx} does not exist in {self.vehicles_dict}.')
            except Exception as e:
                self.logger.warning(f'Client {idx}; Cleanup failed: {e}')  # TODO Determine if this exception can happen and if it needs to be handled.

    def run(self):
        if platform.system() != 'Windows':
            raise NotImplementedError(f'Module {self.type} only works on Windows.')
        self.alive = True
        self.end = False
        PATH_TO_WATCH = Path(config.sites_dir, config.site_name, 'live', 'save')
        FILE_LIST_DIRECTORY = 0x0001
        try:
            try:
                hDir = win32file.CreateFile(
                    str(PATH_TO_WATCH),
                    FILE_LIST_DIRECTORY,
                    win32con.FILE_SHARE_READ,
                    None,
                    win32con.OPEN_EXISTING,
                    win32con.FILE_FLAG_BACKUP_SEMANTICS,
                    None
                )
            except pywintypes.error as e:
                self.logger.critical(f'{e}: {PATH_TO_WATCH}')
                time.sleep(1)
                raise StopModule('Path to watch probably does not exist.')
            try:
                if not self.bind_tcp(timeout=None):
                    raise StopModule(f'Binding to port {self.port} failed!', conn=self.s)
                self.s.listen(5)
                self.logger.debug('Server started.')
            except OSError as e:
                raise StopModule(f'Failed to bind socket: {e}')
            except:
                self.logger.exception('Fatal error:')
                self.alive = False
                self.logger.debug('Thread closed correctly.')
                raise StopModule
            try:
                self.logger.debug('Waiting for connection ...')
                conn, addr = self.accept_tcp(timeout=None)
                self.logger.debug(f'Connection from {addr} accepted.')
            except OSError as e:
                raise StopModule(f'Shutdown flag detected: {e}')
            while True:
                results = set(win32file.ReadDirectoryChangesW(  # Cast to set, so that duplicates of the same event for the same file are skipped
                    hDir,
                    32768,  # This buffer should be able to hold all changed files.
                    True,
                    win32con.FILE_NOTIFY_CHANGE_SIZE,
                ))
                for action, file in results:
                    if action == 3 and Path(file).suffix == '.event':
                        time.sleep(0.1)
                        self.logger.debug(f'Detected {file}.')
                        with open(Path(PATH_TO_WATCH, file), 'rb') as f:
                            event = f.read()
                        length = len(event).to_bytes(length=3, byteorder='big')
                        try:
                            conn.sendall(length)
                            conn.sendall(event)
                        except (ConnectionAbortedError, ConnectionResetError) as e:
                            raise StopModule(e)
                        # Wait for response.
                        response = b''
                        result = None
                        while response == b'':
                            self.logger.debug('Waiting for response ...')
                            response = conn.recv(4)
                            result = struct.unpack('>f', response)[0]
                            self.logger.info(f'Event {Path(file).stem} has predicted weight of {result}.')
                        save_info = {
                            SAVE_PATH: 'ai'
                        }
                        self.write_data(save_info, f'{file}\t{result:.2f}')
                if self.end:
                    raise StopModule('Shutdown flag detected.', conn=conn)
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
        finally:
            if self.s is not None:
                self.s.close()
        self.alive = False
