import copy
import io
import json
import math
import socket
import threading
from collections import deque
from pathlib import Path
from subprocess import check_output
from typing import Any, Dict, Optional, Tuple

import requests
from PIL import Image
from cestel_helpers.aliases import TOMLTable
from lxml import etree
from requests.auth import HTTPDigestAuth

from abstract.module import Module
from consts import CAM_PASS, CAM_USER, MOD_TYPE, RCV_HOST, RCV_OFFSET, RCV_PORT, RCV_SAVE, SAVE_PATH, TEC4_CAMERA_ID, TEC4_DEVICE_STR, TEC4_DEVICE_TYPE, TEC4_REQUEST_TYPE, TEC4_ZONE_ID
from exceptions import NoData, NoPhoto, StopModule
from helpers.helpers import timestamp_to_datetime


class AcquisitionTecdetect4(Module):
    def __init__(self, args):
        self.vehicles_dict = {}
        # Define defaults that may be overwritten by the conf.
        self.camera = 'axis'
        self.camera_id = 1
        self.zone_id = 1
        self.timeout = 5
        self.device_str = '-'.join(check_output(['hostname']).strip().decode().rsplit('-', 2)[1:])
        self.request_type = None
        self.photo_port = None
        self.max_wait_time = 2
        self.host = None
        self.port = None
        self.protocol = 'http://'
        self.uname = 'root'
        self.passw = 'axis'

        # Initialize the module, passing it a list of keys that must exist for it to work properly.
        Module.__init__(self, args, mandatory_keys=(MOD_TYPE, RCV_HOST, RCV_OFFSET, RCV_PORT, RCV_SAVE), optional_keys=('camera_id', 'device_str', 'photo_port', 'request_type', 'timeout', 'zone_id', CAM_USER, CAM_PASS))
        self.type = 'tecdetect4'
        # Define expected keys and what they should map to.
        self.mandatory_keys: Dict[str, Optional[str]] = {
            'DetectionResults': None,
            'EventID': 'id',
            'Filename': 'photo',
            'MsgSize': None,
            'Timestamp': None,
            'ResultCount': None
        }
        self.lanes = [int(k) for k in self.offset.keys()]

    def add_vehicle(self, vehicle, recv_module_name):
        if recv_module_name not in self.vehicles_dict:
            self.vehicles_dict[recv_module_name] = deque()
        self.vehicles_dict[recv_module_name].append(vehicle)

    def get_photo(self, vehicle=None) -> None:
        pass

    def get_mobotix_id(self, vehicle=None) -> Optional[str]:
        num_events_to_parse = 10
        photos_per_sec = 5
        event_list_loc = "/control/event.jpg?output=alarmlist"
        event_img_loc = "/record/events/"

        # Extract relevant data from xml.
        try:
            veh_ts, photo_ts, lane = self.get_photo_data(vehicle, self.max_wait_time)
        except NoPhoto as e:  # This occurs if lane is wrong.
            self.logger.warning(e)
            return

        tsofs = num_events_to_parse / (2 * photos_per_sec)
        evurl = self.protocol + self.camera_ip + event_list_loc
        evurl += "&length=" + str(num_events_to_parse)
        evurl += "&searchbytime_start=" + str(photo_ts + tsofs)  # TODO; do I need to add bias?
        try:
            response = requests.get(evurl, timeout=5)
        except:
            raise NoPhoto('Failed to obtain list of events due to timeout.')
        if not response.ok:
            raise NoPhoto(f'Obtaining list of events failed with status {response.text}: {response.reason}')

        # json array of events
        html = response.content
        json_array = json.loads(html)
        cam_ev_list = list()
        for jso in json_array:
            aiv = jso["alarmimage"]
            tsv = float(jso["timestamp"])
            if len(aiv) != 8:
                continue
            pth = event_img_loc + aiv[0:3] + "/" + aiv[3:6] + "/E00000.jpg"
            cam_ev_list.append((tsv, pth))
        self.logger.debug('Received that many events: ' + str(len(cam_ev_list)))
        if len(cam_ev_list) == 0:
            raise NoPhoto(f'{evurl} response: {html.decode()}')

        mindif = 31
        mini = 0
        for i in range(len(cam_ev_list)):
            dif = math.fabs(photo_ts - cam_ev_list[i][0])
            if dif < mindif:
                mini = i
                mindif = dif
        if mindif > 30:
            raise NoPhoto(f'Failed to match image within 30 seconds.')
        return cam_ev_list[mini][1].split('/', 3)[-1].rsplit('/', 1)[0]

    def parse_data(self, data: Dict[str, Any], vehicle_ts: str, lane: Optional[int] = None, request_photo: bool = True) -> etree.Element:
        """ Attempts to parse received json and convert it to SiWIM friendly format.
        :param data: Received data.
        :param vehicle_ts: Timestamp of the vehicle, which should be used when saving file.
        :param lane: Lane of the vehicle; relevant when saving photos.
        :param request_photo: If true, request the photo and save it.
        """
        node = etree.Element('anpr')
        node.set('source', self.name)
        node.set('type', self.type)

        # Add vehicle timestamp as the key used for matching.
        el = etree.Element('ts')
        el.text = vehicle_ts
        node.append(el)
        # Add other common keys.
        for key, val in self.mandatory_keys.items():
            if val is not None:
                el = etree.Element(val)
                el.text = str(data.get(key))
                node.append(el)

        ts = data['Timestamp']

        anpr_tables_keys: Tuple[str, ...] = ('ObjType', 'ObjRect', 'ObjText', 'ObjScore', 'TextScore', 'ZoneId')
        if 'AnprTables' in data and len(data['AnprTables']) > 0:
            def download_photo(url: str, typ: str, lane: int, text: str = 'notext') -> None:
                """ Downloads photo from url to folder typ."""
                try:
                    response = requests.get(url, timeout=10)
                except (requests.exceptions.ConnectionError, requests.exceptions.ReadTimeout):
                    self.logger.warning(f'Connection error for {photo_loc}.')
                    return
                if response.ok:
                    self.generate_data_dir(typ, vehicle_ts)
                    with open(Path(self.data_dirs[typ], f'{vehicle_ts}_{lane}_p00.jpg'), 'wb') as f:
                        f.write(response.content)
                elif response.status_code == 400:
                    self.logger.error(f'Bad request for {typ}: {text}. Url: {url}.')
                else:
                    self.logger.warning(f'Photo ({typ}: {text}) for {vehicle_ts} not saved: {response.status_code}:{response.text}.')

            photo_list = []
            if request_photo:
                # Get the list of valid photos for this event.
                try:
                    response = requests.get(f'http://{self.host}/tecdetect4-cgi/list.cgi?n=5?ts={ts}', timeout=2)
                except (requests.exceptions.ConnectTimeout, requests.exceptions.ReadTimeout, socket.timeout):
                    raise NoPhoto(f'{ts}: Connection timed out while waiting for list of photos.')
                photo_list = str(response.text).split('<br/>')[:-1]  # This needs to be done because photos have arbitrary indexes.

            for table in data['AnprTables']:  # Go over every detection.
                typ = table['ObjType'].lower()

                # Add relevant information for each detection.
                el = etree.Element('lp' if typ == 'lpr' else 'hgp')
                el.text = table['ObjText']
                node.append(el)

                lane = lane if lane is not None else table['ZoneId']  # If we're getting data from camera, it needs to provide the lane.
                el = etree.Element('lane')
                el.set('source', typ)
                el.text = str(lane)
                node.append(el)

                el = etree.Element('confidence')
                el.set('source', typ)
                el.set('object', str(table['ObjScore']))
                el.set('text', str(table['TextScore']))
                el.text = str(min(table['TextScore'], table['ObjScore']))
                node.append(el)

                # Obtain photo.
                text = table['ObjText']

                # Find the correct photo.
                for photo in photo_list:
                    try:
                        photo_ts, _, _, photo_type, _ = photo.split('_', 4)
                    except ValueError:  # This happens for basic photos, which we don't care for here.
                        continue
                    if photo_ts == ts and photo_type == typ:
                        el = etree.Element('photo')
                        el.set('source', typ)
                        el.text = photo
                        node.append(el)
                        break
                else:
                    if request_photo:  # Only print the warning if requesting of photos is enabled.
                        self.logger.warning(f'No valid photo found for {ts}_{lane} {text}.')
                    continue

                photo_loc = f'http://{self.host}/tecdetect4-cgi/get.cgi?img={photo}'
                download_photo(photo_loc, typ, lane, text=text)

        vehicle_classes_keys: Tuple[str, ...] = ('ObjType', 'ObjRect', 'ObjText', 'ObjScore', 'ZoneId')
        if 'VehicleClasses' in data:
            for table in data['VehicleClasses']:
                el = etree.Element('cls')
                el.set('source', 'tecdetect4')
                el.text = table['ObjType']
                node.append(el)

        for key in data.keys():
            if key not in self.mandatory_keys and key != 'AnprTables' and key != 'VehicleClasses' and key != 'Timeclient':
                self.logger.warning(f'Key {key} not handled')

        return node

    def send_complete_vehicle_downstream(self, vehicle: etree.Element) -> None:
        """ This function goes against currently established rules; rather than send only this module's output, it merges it. """
        for key, mod in self.downstream_modules_dict.items():
            mod.add_vehicle(copy.deepcopy(vehicle), self.name)
        if vehicle.find('wim') is not None:
            self.logger.info(f'Sent {vehicle.find("wim/ts").text}_{vehicle.find("wim/lane").text} downstream.')
        else:
            try:
                self.logger.info(f'Sent {vehicle.find("anpr/ts").text}_{vehicle.find("anpr/lane").text} downstream.')
            except AttributeError:
                self.logger.warning('Attempting to send an invalid auxiliary downstream. Saved to dump')
                self.logger_dump.warning(f'Invalid auxiliary: {etree.tostring(vehicle)}.')

    def camera_thread(self, section: TOMLTable) -> None:
        while True:
            self.throttle()
            if self.end:
                raise StopModule('Shutdown flag detected.')
            self.s.settimeout(self.timeout if section.get(TEC4_REQUEST_TYPE) else None)  # Since the camera can be configured to send only adr photos, set timeout to None for that mode.

            if section.get(TEC4_REQUEST_TYPE):  # If request_type isn't set, it's expected camera will be the one sending data and matching done as usual.
                for key in self.vehicles_dict:
                    if len(self.vehicles_dict[key]) == 0:
                        break
                    sub_veh = self.vehicles_dict[key].popleft()
                    try:
                        veh_ts, photo_ts, lane = self.get_photo_data(sub_veh, self.max_wait_time)
                    except NoPhoto as e:
                        self.logger.warning(e)
                        if sub_veh.find('wim') is not None and sub_veh.find('wim/lane').text in section.get(RCV_OFFSET):  # Only forward the vehicle, if it's on a handled lane.
                            self.send_complete_vehicle_downstream(sub_veh)
                        continue
                    ts = timestamp_to_datetime(photo_ts)

                    size_length = 10
                    # ImageSrc: 1 = image in request, 2 = Image on Jetson locally, 3 = Request image from camera.
                    # CameraId: ID of camera.
                    # DeviceId: ID of system.
                    # DeviceStr: SiWIM ID of the system.
                    # DeviceType: Type of camera; "1" for axis and "2" for mobotix.
                    if section.get(TEC4_DEVICE_TYPE) == 'axis':
                        device_type = 1
                    elif section.get(TEC4_DEVICE_TYPE) == 'mobotix':
                        device_type = 2
                        try:
                            ts = self.get_mobotix_id(sub_veh)
                        except NoPhoto as e:
                            self.logger.error(f'Failed to obtain photo information from camera: {e}')
                    elif section.get(TEC4_DEVICE_TYPE) == 'debug':
                        device_type = 0
                    else:
                        raise NotImplementedError(f'Camera {section.get(TEC4_DEVICE_TYPE)} is not implemented.')
                    data1 = b'{"DetectionRequest":1,"DeviceStr":"' + section.get(TEC4_DEVICE_STR).encode() + b'","DeviceType":' + str(device_type).encode() + b',"CameraId":' + str(
                        section.get(TEC4_CAMERA_ID)).encode() + b',"EventId":1,"ZoneId":' + str(section.get(TEC4_ZONE_ID)).encode() + b',"MaxObjPerZone":1,"Timestamp":"' + ts.encode() + b'","MsgSize":"'
                    data2 = b'","RequestType":' + str(section.get(TEC4_REQUEST_TYPE)).encode() + b',"ImageSrc":3,"ImageType":1}'
                    msg_size = len(data1) + len(data2) + size_length
                    final_msg = data1 + str(msg_size).encode().zfill(size_length) + data2
                    self.logger.debug(f'Sending message {final_msg}.')

                    if self.logger.level == 10:  # Sanity check to make sure the message is valid json. TODO This should be a test.
                        try:
                            json_str = final_msg.decode()
                            json.loads(json_str)
                        except json.decoder.JSONDecodeError as e:
                            self.logger.critical(e)
                            self.logger.debug(final_msg)
                            continue

                    try:
                        self.s.sendall(final_msg)
                    except ConnectionAbortedError:
                        self.send_complete_vehicle_downstream(sub_veh)
                        raise StopModule('Host aborted connection.')

                    try:
                        received_info, buffer = self.acquire_data(self.s, b'', b'{"DetectionResults":1', b'}\x00', 'json', max_buffer=2048, save_info={SAVE_PATH: 'anpr'} if self.save_original else None,
                                                                  exit_on_timeout=True, offset_end=-1, encoding='Windows-1250')
                    except NoData as e:
                        self.send_complete_vehicle_downstream(sub_veh)
                        raise StopModule(e)

                    if buffer != b'\x00' and buffer != b'':
                        self.logger.error(f'There was leftover data of length {len(buffer)} that was not parsed!')
                        self.logger_dump.error(f'Leftover data: {buffer}.')

                    # Make sure the response is for the correct vehicle.
                    if received_info.get('Timeclient') != ts:
                        self.send_complete_vehicle_downstream(sub_veh)
                        if received_info.get('Timeclient') is not None:
                            raise StopModule(f'Incorrect response! Timestamp of sent photo ({ts}) does not match received timestamp ({received_info.get("Timeclient")}).')
                        else:
                            self.logger.error(f'"Timeclient" not found in received data! Matching not possible. Please update {self.host}.')
                            continue

                    if self.camera == 'axis':
                        try:
                            anpr = self.parse_data(received_info, veh_ts, lane)
                        except NoPhoto as e:
                            self.logger.warning(e)
                            continue
                        sub_veh.append(anpr)  # Add anpr data to the received vehicle.
                        self.send_complete_vehicle_downstream(sub_veh)  # Send combined data downstream.
                    elif self.camera == 'mobotix':
                        text = received_info.get('EventID')
                        photo = received_info.get('Filename')
                        url = f'http://{self.host}/tecdetect4-cgi/get.cgi?img={photo}'
                        typ = 'photo'
                        try:
                            response = requests.get(url, timeout=10)
                        except (requests.exceptions.ConnectionError, requests.exceptions.ConnectTimeout, requests.exceptions.ReadTimeout):
                            self.logger.warning(f'Connection error for {veh_ts}.')
                            return
                        if response.ok:
                            self.generate_data_dir(typ, veh_ts)
                            with open(Path(self.data_dirs[typ], f'{veh_ts}_{lane}_p00.jpg'), 'wb') as f:
                                f.write(response.content)
                        elif response.status_code == 400:
                            self.logger.error(f'Bad request for {typ}: {text}. Url: {url}.')
                        else:
                            self.logger.warning(f'Photo ({typ}: {text}) for {veh_ts} not saved: {response.status_code}:{response.text}.')
                    elif self.camera == 'debug':
                        anpr = self.parse_data(received_info, veh_ts, lane, request_photo=False)
                        sub_veh.append(anpr)  # Add anpr data to the received vehicle.
                        self.send_complete_vehicle_downstream(sub_veh)  # Send combined data downstream.
                    else:
                        raise NotImplementedError(f'Camera {self.camera} is not implemented.')
            else:  # There is a camera on the other end.
                try:
                    received_info, buffer = self.acquire_data(self.s, buffer, b'{"DetectionResults":1', b'}\x00', 'json', max_buffer=2048, save_info={SAVE_PATH: 'anpr'} if self.save_original else None,
                                                              exit_on_timeout=True, offset_end=-1, encoding='Windows-1250')
                except NoData as e:
                    raise StopModule(e)

                if self.photo_port:
                    try:
                        url = f'{self.protocol}{self.host}:{self.photo_port}/local/tecdetect3/settings.cgi?name=get&img={received_info.get("Filename")}'
                        self.logger.debug(f'Url: {url}')
                        res = requests.get(url, auth=HTTPDigestAuth(self.uname, self.passw), timeout=5)

                        if not res.ok:
                            self.logger.debug(f'Error message: {res.text}.', stacklevel=2)
                            raise NoPhoto(f'Camera responded with {res.status_code}.')

                        temp_buff = io.BytesIO()
                        self.logger.debug(f'Photo size: {len(res.content)}.', stacklevel=2)
                        temp_buff.write(res.content)
                        temp_buff.seek(0)

                        try:
                            im = Image.open(temp_buff)
                        except:
                            self.logger_dump.error(f'Not valid photo: {res.text}.')
                            raise NoPhoto(f'Failed to parse photo. Saved to dump.')

                        photo_dir = 'photo'
                        self.generate_data_dir(photo_dir, received_info.get('Timestamp'))
                        im.save(Path(self.data_dirs[photo_dir], received_info.get('Filename')))

                        # Save the photo. If saving fails, save a binary blob instead.
                        try:
                            im.save(Path(self.data_dirs[photo_dir], received_info.get('Filename')), 'jpeg', quality="keep", optimize=True)
                        except OSError as e:
                            with open(Path(self.data_dirs[photo_dir], f'{received_info.get("Timestamp")}.truncated'), 'wb') as blob:
                                blob.write(res.content)
                            raise NoPhoto(f'Failed to save {received_info.get("Filename")} as image: {e}. Original data saved as truncated.')
                    except (NoPhoto, requests.exceptions.ReadTimeout, requests.exceptions.ConnectionError) as e:
                        self.logger.warning(f'Failed to save photo for {received_info.get("Timestamp")}: {e}')

                anpr = self.parse_data(received_info, received_info['Timestamp'], request_photo=False)
                veh_node = etree.Element('vehicle')
                veh_node.append(anpr)
                self.send_complete_vehicle_downstream(veh_node)  # Send data downstream.

    def run(self) -> None:
        try:
            if not self.connect_tcp():
                raise StopModule(f'Connection to {self.host}:{self.port} failed.')
            self.logger.info(f'Connection to {self.host}:{self.port} established.')
            buffer = b''

            # if self.request_type:
            #     self.logger.info(f'Running in jetson mode with settings {self.request_type}.')
            # else:
            #     self.logger.info('Running in camera mode.')
            for camera in self.cameras:  # Start a new thread for each camera.
                threading.Thread(target=self.camera_thread, args=(camera,)).start()

        except StopModule as e:
            self.log_stop_module(e)
