import socket
import struct
import time
from collections import deque
from datetime import datetime

from abstract.module import Module
from consts import CAN_BOARD_IDS, CAN_E_IP, CAN_E_PORT, CAN_FWVER, CAN_INTERVAL, CAN_LISTENERS, CAN_RX_IP, CAN_RX_PORT, CAN_TX_IP, CAN_TX_PORT, CAN_UPTIME, MOD_CAN, SAVE_HEADER, SAVE_PATH, SAVE_SUFFIX
from exceptions import StopModule
from helpers.helpers import dict2confstr


class CanModule(Module):
    def __init__(self, args):
        self.ap_uptime = 2
        Module.__init__(self, args, mandatory_keys=(CAN_BOARD_IDS, CAN_E_IP, CAN_E_PORT, CAN_FWVER, CAN_INTERVAL, CAN_LISTENERS, CAN_RX_IP, CAN_RX_PORT, CAN_TX_IP, CAN_TX_PORT), optional_keys=(CAN_UPTIME,))
        self.type = MOD_CAN
        self.vdesc = ["unknown", "analog_12V", "analog_-12V", "analog_5V", "analog_-5V", "digital_12V"]
        self.tdesc = ["unknown", "T1__C", "T2__C", "T3__C", "T4__C", "Ti__C"]
        self.bname = ["unknown", "CAZ", "CAT", "SPS", "BRD4", "BRD5", "BRD6", "BRD7", "SAM1", "SAM2", "SAM3", "SAM4", "SAM5", "SAM6", "SAM7", "SAM8"]
        self.s = None

    def can_msg(self, idx, data):
        if len(data) > 8:
            raise ValueError("invalid data")
        d = bytearray(36)  # initializes elements to zero
        d[1] = 0x24  # = 36
        d[3] = 0x80
        d[21] = len(data)
        d[24:28] = struct.pack('>i', idx)  # > indicates big endian
        d[28:28 + len(data)] = data
        return d

    def brd_cmd(self, board_id, cmd, par1, par2):
        txsock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        txsock.settimeout(1)
        try:
            d = bytearray(8)
            d[0] = board_id
            d[1] = cmd
            d[2:4] = struct.pack('<i', par1)[:2]  # struct.pack returns 4 bytes, but we only care about first 2. < indicates little endian
            d[4:8] = struct.pack('<i', par2)
            txsock.sendto(self.can_msg(0xc0, d), (self.udp_tx_ip, self.udp_tx_port))
        finally:
            txsock.close()

    def send_data(self, data):
        conf_str_data = dict2confstr('mydevice', data)
        sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        try:
            sock.sendto(conf_str_data.encode(), (self.udp_e_ip, self.udp_e_port))
        finally:
            sock.close()

    def run(self):
        try:
            if self.bind_udp(self.udp_rx_ip, self.udp_rx_port):
                command_queue = deque()
                start = time.time()
                polling_start = time.time()
                # implement this so I don't have to wait while testing
                # it won't harm the future either
                polling_interval = 0

                temp_dict = {}
                volt_dict = {}
                send_prompts = True
                while True:
                    self.throttle()
                    if self.end:
                        self.alive = False
                        raise StopModule('Thread closed correctly.', conn=self.s)

                    # poll for data; create requests and send them serially
                    if time.time() - polling_start > polling_interval:
                        # once in a while, send at least one
                        send_prompts = True
                        for board_id in self.board_ids:
                            command_queue.append((board_id, self.ap_uptime, 0, 0))
                            command_queue.append((board_id, self.ap_fwver, 0, 0))
                            polling_interval = self.interval
                        polling_start = time.time()
                    if len(command_queue) > 0 and send_prompts:
                        command = command_queue.popleft()
                        self.brd_cmd(command[0], command[1], command[2], command[3])
                        send_prompts = False
                    if len(command_queue) == 0:
                        send_prompts = True

                    # try to receive, data; except if failed
                    try:
                        data, addr = self.s.recvfrom(64)
                        # whenever we receive something, send more prompts
                        send_prompts = True
                        try:
                            if hasattr(self, 'mk_ip') and hasattr(self, 'mk_port'):
                                self.fw_s.sendto(data, (self.mk_ip, self.mk_port))
                        except:
                            self.logger.exception("Didn't send to mk:")
                    except:
                        continue

                    # read data
                    dlc = data[21]
                    cid = struct.unpack('!I', data[24:28])[0]
                    cdt = data[28:36]
                    # voltage, do nothing
                    if cid == 0x90:
                        vid = data[28]
                        volt_dict[self.vdesc[int(vid)]] = format(struct.unpack('f', data[29:33])[0], '.2f')
                    # temperature
                    elif cid == 0x80:
                        tid = data[28]
                        temp_dict[self.tdesc[int(tid)]] = format(struct.unpack('f', data[29:33])[0], '.2f')
                        temp_dict["ts"] = datetime.now().strftime('%Y-%m-%d-%H-%M-%S')
                    # version
                    elif (cid >= 0xc1) and (cid <= 0xcf):
                        board_id = cdt[0]  # Since we have a bytearray, values are represented as numbers, so cast to int is irrelevant (and not Python3 compatible).
                        par_id = cdt[1]
                        if par_id == 1:  # fwver
                            fwver_major = cdt[3]
                            fwver_minor = cdt[2]
                            ver = f'{fwver_major}.{fwver_minor}'
                            self.set_upstream_info(self.get_name(), self.bname[board_id].lower() + "_firmware_version", ver)
                        elif par_id == 2:  # uptime
                            upt = struct.unpack('I', cdt[4:])[0]
                            self.logger.info(self.bname[board_id] + ' uptime: ' + str(upt))
                    else:
                        self.logger.error('unk ID ' + format(cid, '03x') + ' DATA ' + ' '.join(format(x, '02x') for x in cdt[:dlc]))

                    # forward voltage to downstream (specific analagoue values)
                    for volt_key, val in volt_dict.items():
                        if volt_key != "unknown" and volt_key != "ts":
                            self.set_upstream_info(self.get_name(), volt_key, str(val))

                    # save and forward temperature info once in an 'interval', if relevant
                    if time.time() - start > self.interval and "t" in self.relevant_for_listeners.lower():
                        temps = []
                        for el in self.tdesc[1:]:
                            temps.append(temp_dict.get(el, '-273'))
                        self.write_data({SAVE_PATH: 'temperature', SAVE_HEADER: 'ts\t' + '\t'.join(self.tdesc[1:]), SAVE_SUFFIX: 'tsv'}, temps)
                        self.set_upstream_info(self.get_name(), 'temperature_path', self.data_dirs['temperature'])
                        # -------------------
                        self.send_data(temp_dict)
                        start = time.time()
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
        finally:
            self.s.close()
