import json
from collections import namedtuple
from datetime import datetime, timedelta
from difflib import SequenceMatcher
from pathlib import Path
from time import sleep
from typing import List, Optional

from cestel_helpers.aliases import Element
from lxml import etree

import config
from abstract.postprocessing import PostprocessingModule
from consts import MATCH_FUZZINESS, MATCH_GRACE, MATCH_LOOKBACK, MATCH_MIN_COUNT, MATCH_MODULE, MATCH_OFFSET, TS_FORMAT_STRING
from exceptions import StopModule

Plate = namedtuple('Plate', ['plate', 'ts', 'object'])  # Contains plate as string, timestamp as datetime and etree.Element object


class MatchModule(PostprocessingModule):
    def __init__(self, args):
        self.fixed_offset: int = 0  # Offset in seconds, that fuzziness should be centered at.
        self.fuzziness: int = 60
        self.module: Optional[str] = None
        self.min_match: int = 4
        self.lst_dir: Path = config.sites_dir / config.site_name / 'postproc'
        PostprocessingModule.__init__(self, args, mandatory_keys=(MATCH_MODULE, MATCH_OFFSET), optional_keys=(MATCH_FUZZINESS, MATCH_GRACE, MATCH_LOOKBACK, MATCH_MIN_COUNT))
        self.lst_dir = Path(self.lst_dir)  # Make sure it's a Path object, in case it got set via toml.
        if self.module != 'tecdetect4':
            raise StopModule(f'Module {self.module} is not supported.')

    def find_matching_plates(self, xml_plates, json_plates):
        # Sort both lists by time.
        xml_plates.sort(key=lambda x: x.ts)
        json_plates.sort(key=lambda x: x.ts)

        matches = []
        i = j = 0
        while i < len(xml_plates) and j < len(json_plates):
            xml_plate, xml_ts, vehicle = xml_plates[i]
            json_plate, json_ts, _ = json_plates[j]
            time_diff = (xml_ts - json_ts).total_seconds() - self.fixed_offset

            if abs(time_diff) <= self.fuzziness:
                i += 1  # Variable i needs to be incremented before add_to_xml is called, since it may end the loop.

                if self.add_to_xml(vehicle, xml_plate, json_plate, time_diff, xml_ts):  # If a match is found, continue with the next vehicle.
                    continue

                # Check neighboring elements in the time window.
                for k in range(j + 1, len(json_plates)):
                    plate2_k, time2_k, _ = json_plates[k]
                    time_diff_2 = (xml_ts - time2_k).total_seconds() - self.fixed_offset
                    if abs(time_diff_2) > self.fuzziness:
                        break
                    if self.add_to_xml(vehicle, xml_plate, plate2_k, time_diff_2, xml_ts):  # If a match is found, exit the inner loop and continue with the main loop.
                        break
            elif time_diff < 0:
                i += 1
            else:
                j += 1

        return matches

    def add_match(self, xml_file: Path, lst_file: Path) -> Optional[List[etree.Element]]:
        """ Reads XML and finds all vehicles that can be matched with the LST file.
        :param xml_file: XML file name.
        :param lst_file: LST file we're matching with.
        """
        if not xml_file.is_file():
            self.logger.error(f'{xml_file.name} not found.')
            return None
        if not lst_file.is_file():
            self.logger.error(f'{lst_file.name} not found.')
            return None

        # Parse the XML.
        self.logger.debug(f'Parsing {xml_file.name}.')
        try:
            tree = etree.parse(xml_file)
        except etree.XMLSyntaxError:
            self.logger.error(f'{xml_file.name} is not a valid XML file.')
            return None
        root = tree.getroot()

        # Find all vehicle elements
        vehicles = root.xpath('/swd/site/vehicles/vehicle')

        # Read the lst file.
        with open(lst_file) as f:
            lines = f.readlines()

        # Find all detections.
        self.logger.debug(f'Parsing {lst_file.name}.')
        detections_json: List[Plate] = []
        for line in lines:
            data = json.loads(line)
            ts = datetime.strptime(data['Timestamp'], TS_FORMAT_STRING)
            data = data.get('AnprTables', [])
            for detection in data:
                if detection['ObjType'] == 'LPR':
                    detections_json.append(Plate(detection['ObjText'], ts, None))

        # Iterate over the vehicles to find matching licence plates.
        detections_xml: List[Plate] = []
        for vehicle in vehicles:  # Iterate over all vehicles.
            for plate in [p.text for p in vehicle.xpath('anpr/lp')]:  # Iterate over all plates for the vehicle.
                detections_xml.append(Plate(plate, datetime.strptime(vehicle.find('wim/ts').text, TS_FORMAT_STRING), vehicle))

        self.logger.debug('Matching ...')
        self.find_matching_plates(detections_xml, detections_json)
        self.logger.debug('Finished matching.')

        return vehicles

    def add_to_xml(self, vehicle: Element, xml_plate: str, aux_plate: str, diff: float, xml_ts: datetime) -> bool:
        """ Checks if the plates are a match and adds it to the Element object
        :param vehicle: XML vehicle element.
        :param xml_plate: Plate string of SiWIM detection.
        :param aux_plate: Plate string of auxiliary detection.
        :param diff: Time difference in seconds.
        :param xml_ts: Timestamp of XML timestamp for logging.
        :return: True, if plates match, else False.
        """
        common_chars = sum(block.size for block in SequenceMatcher(None, xml_plate, aux_plate).get_matching_blocks())
        if common_chars >= self.min_match:
            # Log partial matches to info and the rest to debug.
            if xml_plate != aux_plate:
                self.logger.info(f'Matched partial plates {xml_plate} and {aux_plate} at {xml_ts.strftime("%T")} with time difference of {diff:.3f} (offset by {self.fixed_offset}).')
            else:
                self.logger.debug(f'Matched plates {xml_plate} and {aux_plate} at {xml_ts.strftime("%T")} with time difference of {diff:.4f} (offset by {self.fixed_offset}).')
            node = vehicle.find('integration')  # Make sure to reuse node "integration" if it exists.
            if node is None:
                node = etree.Element('integration')
            el = etree.Element('match')
            el.text = '1'
            node.append(el)
            vehicle.append(node)
            return True
        return False

    def run(self) -> None:
        self.alive = True
        self.end = False
        self.logger.debug(f'Started using SWD location {self.swd_dir}.')
        try:
            # Find the first day that needs to be updated.
            for day in range(self.lookback_days, 0, -1):
                date = datetime.now().date() - timedelta(days=day)
                self.logger.debug(f'Checking if {date}.xml needs to be updated ...')
                # If any match tag already exists, the file has been handled already.
                if not (self.lst_dir / f'{date}.lst').is_file():
                    self.logger.warning(f'{date}.lst not found.')
                elif self.swd_to_process(self.swd_dir / f'{date}.xml'):
                    self.logger.info(f'Updating starting with {date}.xml')
                    break
            else:  # If we're not checking past files, use today's date.
                date = datetime.now().date()
                self.logger.debug(f'No valid SWD files found within {self.lookback_days} days.')

            try:
                while True:
                    self.logger.debug('Checking if date changed ...')
                    if self.end:
                        raise StopModule('Shutdown flag detected.')
                    if date != (datetime.now() - timedelta(seconds=self.grace_period)).date():  # Run 30 minutes after midnight, so we can be reasonably sure all the files are in place.
                        self.logger.debug(f'Processing {date}.')
                        xml_path = self.swd_dir / f'{date}.xml'
                        if self.swd_to_process(xml_path):
                            vehicles = self.add_match(xml_path, self.lst_dir / f'{date}.lst')
                            if vehicles:  # Only write the file if add_match was successful.
                                self.logger.info(f'Overwriting {xml_path.name} ...')
                                self.save_swd_file(xml_path, vehicles, set())
                                self.logger.info(f'{xml_path.name} written.')
                        date = date + timedelta(days=1)
                    else:
                        sleep(10)
            except StopModule:
                raise
            except:
                self.logger.exception('Unexpected exception occurred:')
                raise StopModule('Fatal error!')
        except StopModule as e:
            self.log_stop_module(e)
