# Implementation of data acquisition from SiWIM-E.
import copy

from cestel_helpers.exceptions import SiwimError
from cestel_helpers.siwim import Siwim
from lxml import etree

import config
from abstract.module import Module
from consts import MOD_TYPE, RCV_BUFFER, RCV_CYPHER, RCV_HOST, RCV_LANES, RCV_PORT, RCV_SAVE, SAVE_FOOTER, SAVE_HEADER, SAVE_PATH, SAVE_SEPARATOR, SAVE_SUFFIX
from exceptions import NoData, StopModule
from helpers.helpers import get_current_site


class AcquisitionSiwim(Module):
    def __init__(self, args) -> None:
        # Define defaults that may be overwritten by the conf.
        self.buffer = 2048
        self.restart_after_no_data_for = None

        # Initialize the module, passing it a list of keys that must exist for it to work properly and a list of optional keys that are recognized by it.
        Module.__init__(self, args, mandatory_keys=(MOD_TYPE, RCV_HOST, RCV_LANES, RCV_PORT, RCV_SAVE), optional_keys=(RCV_BUFFER, RCV_CYPHER, 'restart_after_no_data_for'))
        self.type = 'siwim'

        # Get SiWIM-E version and site name for use by the application.
        siwim = Siwim(siwim_root=config.sites_dir.parent)
        try:
            config.siwim_e_version = siwim.version_engine
        except SiwimError:
            self.logger.error('SiWIM-E version could not be read!')
        config.site_name = get_current_site(config.siwim_e_conf)

        self.logger.debug(f'Obtained SiWIM-E version {siwim.version_engine} and site name {config.site_name}.')

    def run(self) -> None:
        try:
            self.alive = True
            self.end = False

            if not self.connect_tcp():
                raise StopModule(f'Connection to {self.host}:{self.port} failed.')
            try:
                self.s.settimeout(self.restart_after_no_data_for)  # None is default functionality.
                buffer = b''
                while True:
                    self.throttle()
                    if self.end:
                        raise StopModule('Shutdown flag detected.')
                    try:
                        save_info = {
                            SAVE_PATH: 'wim',
                            SAVE_HEADER: f'<?xml version="1.0" ?>\n<swd version="1"><site><name>{config.site_name}</name><vehicles>\n',
                            SAVE_FOOTER: '</vehicles></site></swd>',
                            SAVE_SUFFIX: 'xml',
                            SAVE_SEPARATOR: ''
                        }
                        received_xml, buffer = self.acquire_data(self.s, buffer, b'<swd ', b'</swd>', 'xml', block_buffer=self.buffer, max_buffer=51200, save_info=save_info if self.save_original else None, exit_on_timeout=True)

                        for vehicle in received_xml.xpath('site/vehicles/vehicle'):
                            # Apply cypher, if it exists.
                            clss = vehicle.xpath('wim/cls')[0].text
                            for key in self.cyphers.keys():
                                key_el = etree.Element(key)
                                key_el.text = self.cyphers[key][clss]
                                vehicle.find('wim').append(key_el)
                            # Send vehicle downstream.
                            for key, mod in self.downstream_modules_dict.items():
                                mod.add_vehicle(copy.deepcopy(vehicle), self.name)
                            self.logger.info(f'Sent {vehicle.xpath("wim/ts")[0].text}_{vehicle.xpath("wim/lane")[0].text} downstream.')
                    except NoData as e:
                        raise StopModule(str(e))
            except StopModule:
                raise
            except:
                self.logger.exception('Fatal error:')
                raise StopModule('Fatal error.')
        except StopModule as e:
            self.log_stop_module(e)
