from collections import deque
from typing import Dict, Optional

import requests
from PIL import ImageFile
from requests.auth import HTTPDigestAuth

from abstract.module import Module
from consts import CAM_HOST, CAM_LANES, CAM_MAX_DIFF, CAM_PASS, CAM_PIC_TYPE, CAM_PORT, CAM_SAMPLING_INTERVAL, CAM_STORY_AFTER, CAM_STORY_BEFORE, CAM_USER, META_CONF, META_MEASUREMENT, META_SPEED, META_TIMESTAMP, META_VEHICLE, MOD_TYPE, RCV_OFFSET
from exceptions import NoPhoto, StopModule
from helpers.helpers import datetime_to_timestamp, get_relevant_tags_from_xml, timestamp_to_datetime

ImageFile.LOAD_TRUNCATED_IMAGES = True  # This is required to save images that are truncated for whatever reason #FIXME Check if this is required by newer pip.


def get_urls(protocol: str, ip: str, port: int) -> Optional[Dict[str, str]]:
    """ This function gets urls based on the first path that exists on the camera. """
    try:
        if requests.get(f'{protocol}{ip}:{port}/local/tecdetect3/settings.cgi', timeout=3).status_code != 404:
            return {'list': f'{protocol}{ip}:{port}/local/tecdetect3/settings.cgi?name=list&', 'get': f'{protocol}{ip}:{port}/local/tecdetect3/settings.cgi?name=get&', 'separator': '&', 'model': 'tecdetect3'}
        elif requests.get(f'{protocol}{ip}:{port}/local/imageretrieve/settings.cgi', timeout=3).status_code != 404:  # V12
            return {'list': f'{protocol}{ip}:{port}/local/imageretrieve/settings.cgi?name=list&', 'get': f'{protocol}{ip}:{port}/local/imageretrieve/settings.cgi?name=get&', 'separator': '&', 'model': 12}
        elif requests.get(f'{protocol}{ip}:{port}/local/image_retrieve/list.cgi', timeout=3) != 404:  # V11
            return {'list': f'{protocol}{ip}:{port}/local/image_retrieve/list.cgi?', 'get': f'{protocol}{ip}:{port}/local/image_retrieve/get.cgi?', 'separator': '?', 'model': 11}
    except requests.exceptions.ConnectTimeout:
        return None
    return None


class CameraAxis(Module):
    def __init__(self, args):
        self.vehicles_dict = {}
        # Define defaults that should be overwritten by the conf.
        self.lanes = []
        self.missed_count = 0
        self.max_time_diff = 0.2
        self.uname = ''
        self.passw = ''
        self.protocol = 'http://'
        self.story = 0
        self.max_wait_time = 0
        self.model = 'N/A'
        # Photo requesting related
        self.story_before = 0
        self.story_after = 0
        self.sampling_interval = 1

        # Initialize the module, passing it a list of keys that must exist for it to work properly.
        Module.__init__(self, args, mandatory_keys=(CAM_HOST, CAM_PIC_TYPE, CAM_PORT, CAM_LANES, CAM_SAMPLING_INTERVAL, CAM_STORY_AFTER, CAM_STORY_BEFORE, MOD_TYPE, RCV_OFFSET), optional_keys=(CAM_PASS, CAM_USER, CAM_MAX_DIFF))
        self.type = 'axis'
        self.urls = get_urls(self.protocol, self.host, self.port)  # This contains list and photo urls for the camera type.

        # Determine correct list and get urls.
        if not self.urls:
            raise StopModule(f'No axis camera found at {self.protocol}{self.host}:{self.port}.')
        self.model = self.urls['model']
        self.logger.debug(f'Using base urls: {self.urls}.')

    def get_photo(self, veh) -> None:
        event_list_loc = self.urls['list']  # add number of events & ts

        # Extract relevant data from xml.
        try:
            veh_ts, photo_ts, lane = self.get_photo_data(veh, self.max_wait_time)
        except NoPhoto as e:  # This occurs if lane is wrong.
            self.logger.warning(e)
            return

        try:
            event_list_loc += f'n={(max(self.story_before, self.story_after) + 1) * 2 * self.sampling_interval + 1}{self.urls["separator"]}ts={timestamp_to_datetime(photo_ts)}'
            response = requests.get(event_list_loc, auth=HTTPDigestAuth(self.uname, self.passw), timeout=10)
        except (requests.exceptions.ConnectionError, requests.exceptions.ReadTimeout):
            raise NoPhoto(f'Connection to {event_list_loc} refused!')
        # list of picture timestamps
        con = response.content
        lines = str(con.decode(encoding='utf-8' if self.model != 'tecdetect3' else 'Windows-1250')).split('<br/>')[:-1]
        cam_ev_list = {}
        for line in lines:
            try:
                if self.model == 'tecdetect3':  # Tecdetect3 sends other information with the photo.
                    line = line.split('_')[0].split('.')[0]
                tsv = datetime_to_timestamp(line)
            except ValueError:
                raise NoPhoto(f'Response for {event_list_loc} is "{line}", which is not a valid date.')
            cam_ev_list[tsv] = self.urls['get'] + f'img={line}.{"jpg" if self.model == "tecdetect3" else "jpeg"}'
        if len(cam_ev_list) == 0:
            self.logger.debug(f'Event query response: {con.decode()}')
        metadata = get_relevant_tags_from_xml(veh.find('wim'), (META_CONF, META_MEASUREMENT, META_SPEED, META_TIMESTAMP, META_VEHICLE))
        self.download_and_save(cam_ev_list, veh_ts, photo_ts, lane, True, metadata=metadata)

    def add_vehicle(self, vehicle, recv_module_name):
        if recv_module_name not in self.vehicles_dict:
            self.vehicles_dict[recv_module_name] = deque()
        self.vehicles_dict[recv_module_name].append(vehicle)

    def run(self) -> None:
        self.alive = True
        self.end = False

        try:
            self.camera_loop()
        except StopModule as e:
            self.log_stop_module(e)
        except:
            self.logger.exception('Fatal error:')
            self.alive = False
